#ifndef PhysLiteToOpenData_PhysLiteToOpenData_H
#define PhysLiteToOpenData_PhysLiteToOpenData_H

//***************************************
// Algorithm for writing ntuple branches
// for the open data ntuple maker
//***************************************

/// Base class for our algorithm
#include <AnaAlgorithm/AnaAlgorithm.h>

/// For properties of our algorithm that we'd like to set
#include <AsgTools/PropertyWrapper.h>

/// Selection / accessor helpers
#include <SelectionHelpers/SelectionReadAccessorBits.h>

/// EDM definitions for our event data
#include <xAODJet/JetContainer.h>
#include <xAODEgamma/ElectronContainer.h>
#include <xAODMuon/MuonContainer.h>
#include <xAODTau/TauJetContainer.h>
#include <xAODMissingET/MissingETContainer.h>
#include <xAODEgamma/PhotonContainer.h>
#include <xAODEventInfo/EventInfo.h>
#include <xAODTruth/TruthParticleContainer.h>

/// Basic tool handle
#include "AsgTools/AnaToolHandle.h"
/// Interface of tool for accessing physics modelling group-provided cross-sections, k-factors, etc
#include "PMGAnalysisInterfaces/IPMGCrossSectionTool.h"
/// Interface of tool for accessing jet energy scale uncertainties
#include "JetCPInterfaces/ICPJetUncertaintiesTool.h"

/// ROOT and std library includes
#include <TTree.h>
#include <vector>

/// For messaging
#include <AsgMessaging/MessageCheck.h>

/// Class for holding event information
/// Created and destroyed during execute
class Event
{
 public:
  // Basic constructor and destructor
  Event ()
   : eventInfo(nullptr)
   , jets(nullptr)
   , electrons(nullptr)
   , muons(nullptr)
   , taus(nullptr)
   , photons(nullptr)
   , allMET(nullptr)
   , largeRJets(nullptr)
   , truthJets(nullptr)
   , truthElec(nullptr)
   , truthMuon(nullptr)
   , truthTaus(nullptr)
   , truthPhot(nullptr)
   , truthMET(nullptr)
  {};

  ~Event(){
    if( eventInfo ){ delete eventInfo; eventInfo = nullptr; }
    if( jets ){ delete jets; jets = nullptr; }
    if( electrons ){ delete electrons; electrons = nullptr; }
    if( muons ){ delete muons; muons = nullptr; }
    if( taus ){ delete taus; taus = nullptr; }
    if( photons ){ delete photons; photons = nullptr; }
    if( allMET ){ delete allMET; allMET = nullptr; }
    if( largeRJets ){ delete largeRJets; largeRJets = nullptr; }
    if( truthJets ){ delete truthJets; truthJets = nullptr; }
    if( truthElec ){ delete truthElec; truthElec = nullptr; }
    if( truthMuon ){ delete truthMuon; truthMuon = nullptr; }
    if( truthTaus ){ delete truthTaus; truthTaus = nullptr; }
    if( truthPhot ){ delete truthPhot; truthPhot = nullptr; }
    if( truthMET ){ delete truthMET; truthMET = nullptr; }
  }

  /// All member variables public so they can be directly accessed / set
  const xAOD::EventInfo * eventInfo;         ///! Event info (event-level metadata)
  const xAOD::JetContainer* jets;            ///! Jets
  const xAOD::ElectronContainer* electrons;  ///! Electrons
  const xAOD::MuonContainer* muons;          ///! Muons
  const xAOD::TauJetContainer* taus;         ///! Tau jets
  const xAOD::PhotonContainer* photons;      ///! Photons
  const xAOD::MissingETContainer* allMET;    ///! Missing transverse momentum
  const xAOD::JetContainer* largeRJets;      ///! Large-radius jets
  const xAOD::JetContainer* truthJets;           ///! Truth Jets
  const xAOD::TruthParticleContainer* truthElec; ///! Truth electrons
  const xAOD::TruthParticleContainer* truthMuon; ///! Truth muons
  const xAOD::TruthParticleContainer* truthTaus; ///! Truth taus
  const xAOD::TruthParticleContainer* truthPhot; ///! Truth photons
  const xAOD::MissingETContainer* truthMET;      ///! Truth missing transverse momentum
};

/// The actual algorithm we use for looping over the data and filling our ntuple
/// Based significantly on the ATLAS analysis tutorial
class PhysLiteToOpenData : public EL::AnaAlgorithm
{
 public:
  // Standard constructor and destructor
  PhysLiteToOpenData (const std::string& name, ISvcLocator* pSvcLocator);
  ~PhysLiteToOpenData ();

  // Functions inherited from Algorithm
  virtual StatusCode initialize () override; ///! Runs once for setup of variables and branches
  virtual StatusCode execute () override;    ///! Runs every event for filling of variables
  virtual StatusCode finalize () override;   ///! Runs at the end of the job for cleanup

  /// Helper for filling the Event (m_event) with the relevant collections for the current event
  StatusCode fillEvent ();

 private:
  /// Tool for extracting cross section, k-factor, and filter efficiency from central database
  asg::AnaToolHandle<PMGTools::IPMGCrossSectionTool> m_PMGCrossSectionTool;

  // Configuration, and any other types of variables go here.
  // Takes the form m_variable{this, "property name", "default value", "Help string"}
  Gaudi::Property<std::string> m_dataType {this, "dataType", "data", "DataType (mc, afii, data)"};

  /// Accessors for various selection cuts
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_isolated_electron_loose;  ///! Electron (Loose+Blayer ID) passes loose isolation
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_isolated_electron_tightiso; ///! Electron (Loose+Blayer ID) passes tight isolation
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_selectLikelihood_electron_loose; ///! Electron passes loose ID criteria
  // Muon decorations
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_good_muon_loose; ///! Muon passes loose ID criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_good_muon_medium; ///! Muon passes medium ID criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_good_muon_tight; ///! Muon passes tight ID criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_isolated_muon_medium; ///! Muon (medium ID) passes loose isolation criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_isolated_muon_tightiso; ///! Muon (medium ID) passes tight isolation criteria
  // Photon decorations
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_isolated_photon_tightiso;  ///! Photon passes tight isolation criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_isolated_photon_loose; ///! Photon passes loose isolation criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_selectEM_photon_loose; ///! Photon passes loose ID criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_selectEM_photon_medium; ///! Photon passes medium ID criteria
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_selectEM_photon_tight; ///! Photon passes tight ID criteria
  // Tau decorations
  std::unique_ptr<CP::ISelectionReadAccessor> accEx_selected_tau_tight; ///! Tau passes tight ID criteria

  /// Current event
  Event * m_event = nullptr;

  /// Weights - These scale factors and weights only apply in MC simulation
  /// They are here to help correct the MC to better match the data
/// Weights
  float m_ScaleFactor_FTAG; ///! The flavour tagging weight
  float m_ScaleFactor_PILEUP; ///! The pileup weight
  float m_ScaleFactor_MLTRIGGER; ///! The total scale factor for multilepton triggers
  float m_ScaleFactor_ELE;  ///! The total scale factor for selected electrons
  float m_ScaleFactor_MUON; ///! The total scale factor for selected muons    
  float m_ScaleFactor_BTAG; ///! The total b-tagging scale factor for all jets
  float m_ScaleFactor_JVT;  ///! The total scale factor for the Jet Vertex Tagger (JVT) 
  float m_ScaleFactor_PHOTON; ///! The total scale factor for selected photons
  float m_ScaleFactor_TAU; ///! The total scale factor for selected taus
  float m_ScaleFactor_LepTRIGGER; ///! The total scale factor for single lepton triggers
  float m_ScaleFactor_ElTRIGGER; ///! The total scale factor for single electron triggers
  float m_ScaleFactor_MuTRIGGER; ///! The total scale factor for single muon triggers
  float m_ScaleFactor_TauTRIGGER; ///! The total scale factor for single tau triggers
  float m_ScaleFactor_DiTauTRIGGER; ///! The total scale factor for di-tau triggers
  float m_mcWeight;  ///! The generator weight for this sample

  float m_TriggerMatch_Dilepton;

  /// Event level variables
  /// Trigger decisions: Did this event pass the electron (E), muon (M), MET, photon (P), Tau (T), di-Tau (DT), multi-lepton (ML), di-electron (DE) or di-muon (DM) trigger combinations  
  bool m_trigE, m_trigM, m_trigMET, m_trigP, m_trigT, m_trigDT, m_trigML, m_trigDE, m_trigDM;

  /// Jet variables
  int m_jet_n = 0;                                  ///! Number of jets
  std::vector<float> *m_jet_pt = nullptr;           ///! Jet transverse momentum
  std::vector<float> *m_jet_eta = nullptr;          ///! Jet pseudo-rapidity
  std::vector<float> *m_jet_phi = nullptr;          ///! Jet phi
  std::vector<float> *m_jet_e = nullptr;            ///! Jet energy
  std::vector<int> *m_jet_btag_quantile = nullptr;  ///! Jet b-tagging quantile (how likely is this a b-jet)
  std::vector<bool> *m_jet_jvt = nullptr;           ///! Jet JVT selection (was this jet associated with the primary vertex)

  /// Systematic uncertainties for jets
  std::vector<float> *m_jet_pt_jer1 = nullptr;     ///! Jet transverse momentum
  std::vector<float> *m_jet_pt_jer2 = nullptr;   ///! Jet transverse momentum

  /// Tool for jet energy scale uncertainties
  asg::AnaToolHandle<ICPJetUncertaintiesTool> m_jetUncertaintiesTool;

  /// LargeRJets variables
  int m_largeRJet_n =0;                             ///! Number of large-R jets
  std::vector<float> *m_largeRJet_pt = nullptr;     ///! Large-R jet transverse momentum
  std::vector<float> *m_largeRJet_eta = nullptr;    ///! Large-R jet pseudo-rapidity
  std::vector<float> *m_largeRJet_phi = nullptr;    ///! Large-R jet phi
  std::vector<float> *m_largeRJet_e = nullptr;      ///! Large-R jet energy
  std::vector<float> *m_largeRJet_m = nullptr;      ///! Large-R jet mass (these are heavy, so mass is important)
  std::vector<float> *m_largeRJet_D2 = nullptr;     ///! Large-R jet D2 (variable useful for identifying W/Z/h boson-induced jets)

  /// Lepton variables
  /// Electrons and muons get summarized in lep_* variables
  /// to distinguish them, m_lep_type gets added (11 for electrons, 13 for muons)
  int m_lep_n = 0;                                  ///! Number of leptons (total)
  std::vector<int> *m_lep_type = nullptr;           ///! Type of lepton (11 for electron, 13 for muon)
  std::vector<float> *m_lep_pt = nullptr;           ///! Lepton transverse momentum
  std::vector<float> *m_lep_eta = nullptr;          ///! Lepton pseudo-rapidity
  std::vector<float> *m_lep_phi = nullptr;          ///! Lepton phi
  std::vector<float> *m_lep_e = nullptr;            ///! Lepton energy
  std::vector<int> *m_lep_charge = nullptr;         ///! Lepton charge (+1/-1)

  std::vector<float> *m_lep_ptvarcone30 = nullptr;  ///! Lepton track-based isolation (variable radius cone)
  std::vector<float> *m_lep_topoetcone20 = nullptr; ///! Lepton calo-based isolation (fixed radius of 0.2)
  std::vector<float> *m_lep_z0 = nullptr;           ///! Lepton z0 with respect to the primary vertex
  std::vector<float> *m_lep_d0 = nullptr;           ///! Lepton d0 (impact parameter) with respect to the primary vertex
  std::vector<float> *m_lep_d0sig = nullptr;        ///! Lepton d0 significance (d0 / d0 uncertainty)

  std::vector<bool> *m_lep_isTightID = nullptr; ///! Did the lepton pass tight ID criteria 
  std::vector<bool> *m_lep_isMediumID = nullptr; ///! Did the lepton pass medium ID criteria
  std::vector<bool> *m_lep_isLooseID = nullptr; ///! Did the lepton pass loose ID criteria
  std::vector<bool> *m_lep_isTightIso = nullptr; ///! Did the lepton pass tight isolation criteria
  std::vector<bool> *m_lep_isLooseIso = nullptr; ///! Did the lepton pass loose isolation criteria

  std::vector<bool> *m_lep_isTrigMatched = nullptr; ///! Did the lepton match any of the triggers 
  
  /// Tau variables
  int m_tau_n = 0;                                  ///! Number of taus
  std::vector<float> *m_tau_pt = nullptr;           ///! Tau transverse momentum
  std::vector<float> *m_tau_eta = nullptr;          ///! Tau pseudo-rapidity
  std::vector<float> *m_tau_phi = nullptr;          ///! Tau phi
  std::vector<float> *m_tau_e = nullptr;            ///! Tau energy
  std::vector<float> *m_tau_charge = nullptr;       ///! Tau charge (+/-1)
  std::vector<int> *m_tau_nTracks = nullptr;        ///! Number of charged tracks associated with the tau (1 or 3 almost always)
  std::vector<bool> *m_tau_isTight = nullptr;       ///! Did the tau pass the tight selection criteria
  std::vector<float> *m_tau_RNNJetScore = nullptr;  ///! Output score of the RNN used to separate taus and hadronic jets
  std::vector<float> *m_tau_RNNEleScore = nullptr;  ///! Output score of the RNN used to separate taus and electrons

  /// Photon variables
  int m_photon_n = 0;                                   ///! Number of photons
  std::vector<float> *m_photon_pt = nullptr;            ///! Photon transverse momentum                                 
  std::vector<float> *m_photon_eta = nullptr;           ///! Photon pseudo-rapidity                        
  std::vector<float> *m_photon_phi = nullptr;           ///! Photon phi                            
  std::vector<float> *m_photon_e = nullptr;             ///! Photon energy                                        
  std::vector<float> *m_photon_ptcone20 = nullptr;      ///! Photon track-based isolation (fixed radius of 0.2)                                     
  std::vector<float> *m_photon_topoetcone40 = nullptr;  ///! Photon calorimeter-based isolation (fixed radius of 0.4)
  std::vector<bool> *m_photon_isLooseID = nullptr;      ///! Did the photon pass the loose ID criteria
  std::vector<bool> *m_photon_isTightID = nullptr;      ///! Did the photon pass the tight ID criteria
  std::vector<bool> *m_photon_isLooseIso = nullptr;     ///! Did the photon pass the loose isolation criteria
  std::vector<bool> *m_photon_isTightIso = nullptr;     ///! Did the photon pass the tight isolation criteria

  // Truth variables
  int m_truth_jet_n = 0;                                ///! Number of truth jets
  std::vector<float> *m_truthJet_pt = nullptr;          ///! Truth jet transverse momentum
  std::vector<float> *m_truthJet_eta = nullptr;         ///! Truth jet pseudo-rapidity
  std::vector<float> *m_truthJet_phi = nullptr;         ///! Truth jet phi
  std::vector<float> *m_truthJet_m = nullptr;           ///! Truth jet mass
  int m_truth_elec_n = 0;                               ///! Number of truth electrons
  std::vector<float> *m_truthElec_pt = nullptr;         ///! Truth electron transverse momentum
  std::vector<float> *m_truthElec_eta = nullptr;        ///! Truth electron pseudo-rapidity
  std::vector<float> *m_truthElec_phi = nullptr;        ///! Truth electron phi
  int m_truth_muon_n = 0;                               ///! Number of truth muons
  std::vector<float> *m_truthMuon_pt = nullptr;         ///! Truth muon transverse momentum
  std::vector<float> *m_truthMuon_eta = nullptr;        ///! Truth muon pseudo-rapidity
  std::vector<float> *m_truthMuon_phi = nullptr;        ///! Truth muon phi
  int m_truth_tau_n = 0;                                ///! Number of truth taus
  std::vector<float> *m_truthTau_pt = nullptr;          ///! Truth tau transverse momentum
  std::vector<float> *m_truthTau_eta = nullptr;         ///! Truth tau pseudo-rapidity
  std::vector<float> *m_truthTau_phi = nullptr;         ///! Truth tau phi
  int m_truth_photon_n = 0;                             ///! Number of truth photons
  std::vector<float> *m_truthPhot_pt = nullptr;         ///! Truth photon transverse momentum
  std::vector<float> *m_truthPhot_eta = nullptr;        ///! Truth photon pseudo-rapidity
  std::vector<float> *m_truthPhot_phi = nullptr;        ///! Truth photon phi
  float m_truth_met;                                    ///! Magnitude of truth MET
  float m_truth_met_phi;                                ///! Angle of truth MET

  /// MET variables
  float m_met;                                      ///! Missing transverse momentum magnitude
  float m_met_phi;                                  ///! Missing transverse momentum angle (phi)
  float m_met_mpx;                                  ///! x-component of missing transverse momentum
  float m_met_mpy;                                  ///! y-component of missing transverse momentum
  
  // Cross section run/MC information
  unsigned int m_channelNumber;         ///! Dataset identifier for this sample (unique integer ID for the dataset)
  unsigned int m_runNumber;             ///! Run number in data
  unsigned int m_randomRunNumber;       ///! A run number assigning the MC to the right data taking year (2015 or 2016)
  unsigned long long m_eventNumber;     ///! Unique event number for each run number in data
  
  float m_xsec;                         ///! Cross-section for this sample (usually taken from the generator directly)                                    
  float m_filteff;                      ///! Filter efficiency for this sample (e.g. if a lepton filter was applied, what fraction of events was retained)
  float m_kfac;                         ///! Higher-order cross-section correction factor
  
  void initJetInfo(TTree * mytree);
  void initLeptonInfo(TTree * mytree);
  void initPhotonInfo(TTree * mytree);
  void initTauInfo(TTree * mytree);
  void initTruthInfo(TTree * mytree);

  /// Helper functions to reset the branches for the different physics objects, called at the start of each event execute()
  void resetJetInfo();
  void resetLeptonInfo();
  void resetPhotonInfo();
  void resetTauInfo();
  void resetTruthInfo();

  /// Helper functions to fill the branches for the different physics objects, called in execute()
  StatusCode fillJetInfo();
  StatusCode fillLeptonInfo();
  StatusCode fillPhotonInfo();
  StatusCode fillTauInfo();
  StatusCode fillTruthInfo();

  /// Helper functions to clean up the branch variables for the different physics objects, called in the destructor
  void cleanupJetInfo();
  void cleanupLeptonInfo();
  void cleanupPhotonInfo();
  void cleanupTauInfo();
  void cleanupTruthInfo();
};

#endif
