/// Header for the class
#include <PhysLiteToOpenData/PhysLiteToOpenData.h>

/// Initialize all our jet branches and variables (called in initialize)
void PhysLiteToOpenData::initJetInfo(TTree * mytree){

  /// Event-level b-tagging scale factor
  m_ScaleFactor_BTAG = 1.0;
  mytree->Branch("ScaleFactor_BTAG",&m_ScaleFactor_BTAG);
  mytree->Branch("ScaleFactor_JVT",&m_ScaleFactor_JVT);

  // Jet variables (see header for details)
  m_jet_n = 0;
  m_jet_pt  = new std::vector<float>();
  m_jet_eta = new std::vector<float>();
  m_jet_phi = new std::vector<float>();
  m_jet_e   = new std::vector<float>();
  m_jet_btag_quantile = new std::vector<int>();
  m_jet_jvt = new std::vector<bool>();

  mytree->Branch("jet_n", &m_jet_n);
  mytree->Branch("jet_pt", &m_jet_pt);
  mytree->Branch("jet_eta", &m_jet_eta);
  mytree->Branch("jet_phi", &m_jet_phi);
  mytree->Branch("jet_e", &m_jet_e);
  mytree->Branch("jet_btag_quantile", &m_jet_btag_quantile);
  mytree->Branch("jet_jvt", &m_jet_jvt);

  // Large-radius jet variables (see header for details)
  m_largeRJet_n = 0;
  m_largeRJet_pt  = new std::vector<float>();
  m_largeRJet_eta = new std::vector<float>();
  m_largeRJet_phi = new std::vector<float>();
  m_largeRJet_e   = new std::vector<float>();
  m_largeRJet_m   = new std::vector<float>();

  mytree->Branch("largeRJet_n", &m_largeRJet_n);
  mytree->Branch("largeRJet_pt", &m_largeRJet_pt);
  mytree->Branch("largeRJet_eta", &m_largeRJet_eta);
  mytree->Branch("largeRJet_phi", &m_largeRJet_phi);
  mytree->Branch("largeRJet_e", &m_largeRJet_e);
  mytree->Branch("largeRJet_m", &m_largeRJet_m);
  mytree->Branch("largeRJet_D2", &m_largeRJet_D2);

  // Systematic variations (only filled for MC)
  // Extra variables for systematic uncertainties
  m_jet_pt_jer1  = new std::vector<float>();
  m_jet_pt_jer2  = new std::vector<float>();
  mytree->Branch("jet_pt_jer1", &m_jet_pt_jer1);
  mytree->Branch("jet_pt_jer2", &m_jet_pt_jer2);
  
}

/// Reset all branch variables for jets (called in execute)
void PhysLiteToOpenData::resetJetInfo(){

  m_ScaleFactor_BTAG = 1.0;
  m_ScaleFactor_JVT = 1.0;
  
  m_jet_n = 0;
  m_jet_pt->clear();
  m_jet_eta->clear();
  m_jet_phi->clear();
  m_jet_e->clear();
  m_jet_btag_quantile->clear();
  m_jet_jvt->clear();
  
  m_largeRJet_n = 0;
  m_largeRJet_pt->clear();
  m_largeRJet_eta->clear();
  m_largeRJet_phi->clear();
  m_largeRJet_e->clear();
  m_largeRJet_m->clear();
  m_largeRJet_D2->clear();

  // Systematic variations (only filled for MC)
  // Reset systematic variation branches
  m_jet_pt_jer1->clear();
  m_jet_pt_jer2->clear();
  
}

/// Fill event information into jet variables and branches (called in execute)
StatusCode PhysLiteToOpenData::fillJetInfo(){

  // An extra piece for systematic variations
  // We will work with two systematic uncertainty variations for jet energy resolution.
  // These are "effective nuisance parameters" (which means they are mash-ups of various
  // uncertainty components). If you want a full list, you can get one from the tools:
  //CP::SystematicSet testSet = m_jetUncertaintiesTool->recommendedSystematics();
  //for (const auto & asyst : testSet){
  //  std::cout << "Variation " << asyst.name() << std::endl;
  //}

  // Set the uncertainties themselves. These can be static - they don't change.
  // They are reasonably small, so we don't have to worry about avoiding this if we
  // are running on data
  const static CP::SystematicSet jer1("JET_JER_EffectiveNP_1__1up");
  const static CP::SystematicSet jer2("JET_JER_EffectiveNP_2__1up");

  /// First a loop through the small-R jets
  for (const xAOD::Jet* jet : *m_event->jets){
    /// Ensure that the jet passes the overlap-removal pre-selection, overlap-removal selection, and kinematic criteria
    if( !jet->auxdata<char>("preselectOR") ) continue;
    if( !jet->auxdata<char>("passesOR") ) continue;
    if( !jet->auxdata<unsigned int>("selectPtEta") ) continue;

    /// Store the jet kinematics (convert to GeV)
    m_jet_pt->push_back(jet->pt()*0.001);
    m_jet_eta->push_back(jet->eta());
    m_jet_phi->push_back(jet->phi());
    m_jet_e->push_back(jet->e()*0.001);

    /// Get the b-tagging information
    m_jet_btag_quantile->push_back(jet->auxdata<int>("ftag_quantile_DL1dv01_Continuous"));

    /// Number of jet counter
    m_jet_n++;

    /// In case the JVT selection is on the jet. This is a decoration
    /// which is added to the jets as long as one does not set runJvtSelection to False
    /// (default is True) in the jet configuration.
    if( jet->isAvailable<char>("jvt_selection") ){
      m_jet_jvt->push_back(bool(jet->auxdata<char>("jvt_selection")));
    }else{ /// In case runJvtSelection is set to False (default is True)
      m_jet_jvt->push_back(false);
    }
    
    /// For the case of MC simulation, we keep track of the event-level b-tagging scale factor
    /// which is just the product of the scale factors for all the individual jets
    if( m_dataType != "data" ){ 
      m_ScaleFactor_BTAG *= jet->auxdata<float>("ftag_effSF_DL1dv01_Continuous_NOSYS");
      if(jet->auxdata<float>("jvt_effSF_NOSYS") != -1)
        m_ScaleFactor_JVT *= jet->auxdata<float>("jvt_effSF_NOSYS"); 
    }

    /// Also for MC simulation, we can check on systematic variations
    if( m_dataType != "data" ){
      // Tell the tool what uncertainty we want it to apply
      if (m_jetUncertaintiesTool->applySystematicVariation(jer1) != StatusCode::SUCCESS) {
        std::cout << "Warning: CP tool failed on uncertainty 1..." << std::endl;
      }
      xAOD::Jet * jet1 = nullptr;
      if (m_jetUncertaintiesTool->correctedCopy(*jet,jet1) != CP::CorrectionCode::Ok) {
        std::cout << "Warning: CP tool failed..." << std::endl;
      }
      /// Store the jet kinematics (convert to GeV)
      m_jet_pt_jer1->push_back( jet1->pt()*0.001 );
      delete jet1;

      // Now set the tool to use the next uncertainty that we're interested in
      if (m_jetUncertaintiesTool->applySystematicVariation(jer2) != StatusCode::SUCCESS) {
        std::cout << "Warning: CP tool failed ..." << std::endl;
      }
      xAOD::Jet * jet2 = nullptr;
      if (m_jetUncertaintiesTool->correctedCopy(*jet,jet2) != CP::CorrectionCode::Ok) {
        std::cout << "Warning: CP tool failed..." << std::endl;
      }
      /// Store the jet kinematics (convert to GeV)
      m_jet_pt_jer2->push_back( jet2->pt()*0.001 );
      delete jet2;

    } // End of check for data vs MC

  }
  
  /// Now a loop through the large-R jets
  for (const xAOD::Jet* jet : *m_event->largeRJets){
    /// Ensure that the jet passes the overlap-removal pre-selection, overlap-removal selection, and kinematic criteria
    if( !jet->auxdata<char>("preselectOR") ) continue;
    if( !jet->auxdata<char>("passesOR") ) continue;
    if( !jet->auxdata<unsigned int>("selectPtEtaMine") ) continue;

    /// Store the jet kinematics (convert to GeV)
    m_largeRJet_pt->push_back(jet->pt()*0.001);
    m_largeRJet_eta->push_back(jet->eta());
    m_largeRJet_phi->push_back(jet->phi());
    m_largeRJet_e->push_back(jet->e()*0.001);
    m_largeRJet_m->push_back(jet->m()*0.001);

    /// Get the tagging variable D2
    m_largeRJet_D2->push_back(jet->auxdata< float >("D2"));

    /// Number of large-R jets counter
    m_largeRJet_n++;
  } // Done with the large-R jets!

  return StatusCode::SUCCESS;
}

/// Cleanup function - delete all our vector branches (called in destructor)
void PhysLiteToOpenData::cleanupJetInfo(){

  delete m_jet_pt;
  delete m_jet_eta;
  delete m_jet_phi;
  delete m_jet_e;
  delete m_jet_btag_quantile;
  delete m_jet_jvt;

  delete m_largeRJet_pt;
  delete m_largeRJet_eta;
  delete m_largeRJet_phi;
  delete m_largeRJet_e;
  delete m_largeRJet_m;
  delete m_largeRJet_D2;

  // Systematic variations (only filled for MC) 
  delete m_jet_pt_jer1;
  delete m_jet_pt_jer2;
  
}
