/// Header for the class
#include <PhysLiteToOpenData/PhysLiteToOpenData.h>

/// Helpers for calculating z0 / d0 w.r.t. primary vertex
#include <xAODTracking/TrackParticlexAODHelpers.h>


/// Initialize all our lepton branches and variables (called in initialize)
void PhysLiteToOpenData::initLeptonInfo(TTree * mytree){

  // Event-level electron and muon scale factors
  m_ScaleFactor_ELE = 1.0;
  m_ScaleFactor_MUON = 1.0;
  m_ScaleFactor_LepTRIGGER = 1.0;

  mytree->Branch("ScaleFactor_ELE",&m_ScaleFactor_ELE);
  mytree->Branch("ScaleFactor_MUON",&m_ScaleFactor_MUON);
  mytree->Branch("ScaleFactor_LepTRIGGER",&m_ScaleFactor_LepTRIGGER);

  mytree->Branch("ScaleFactor_MuTRIGGER",&m_ScaleFactor_MuTRIGGER);
  mytree->Branch("ScaleFactor_ElTRIGGER",&m_ScaleFactor_ElTRIGGER);

  // Lepton variables (see header for details)
  m_lep_n = 0;
  m_lep_type = new std::vector<int>();
  m_lep_pt  = new std::vector<float>();
  m_lep_eta = new std::vector<float>();
  m_lep_phi = new std::vector<float>();
  m_lep_e   = new std::vector<float>();
  m_lep_charge = new std::vector<int>();

  m_lep_ptvarcone30 = new std::vector<float>();
  m_lep_topoetcone20 = new std::vector<float>();
  m_lep_z0 = new std::vector<float>();
  m_lep_d0 = new std::vector<float>();
  m_lep_d0sig = new std::vector<float>();

  m_lep_isTightID = new std::vector<bool>();
  m_lep_isMediumID = new std::vector<bool>();
  m_lep_isLooseID = new std::vector<bool>();
  m_lep_isTightIso = new std::vector<bool>();
  m_lep_isLooseIso = new std::vector<bool>();
  
  mytree->Branch("lep_n", &m_lep_n);
  mytree->Branch("lep_type", &m_lep_type);
  mytree->Branch("lep_pt", &m_lep_pt);
  mytree->Branch("lep_eta", &m_lep_eta);
  mytree->Branch("lep_phi", &m_lep_phi);
  mytree->Branch("lep_e", &m_lep_e);
  mytree->Branch("lep_charge", &m_lep_charge);
  
  mytree->Branch("lep_ptvarcone30", &m_lep_ptvarcone30);
  mytree->Branch("lep_topoetcone20", &m_lep_topoetcone20);
  mytree->Branch("lep_z0", &m_lep_z0);
  mytree->Branch("lep_d0", &m_lep_d0);
  mytree->Branch("lep_d0sig", &m_lep_d0sig);

  mytree->Branch("lep_isTightID", &m_lep_isTightID);
  mytree->Branch("lep_isMediumID", &m_lep_isMediumID);
  mytree->Branch("lep_isLooseID", &m_lep_isLooseID);
  mytree->Branch("lep_isTightIso", &m_lep_isTightIso);
  mytree->Branch("lep_isLooseIso", &m_lep_isLooseIso);

  mytree->Branch("lep_isTrigMatched",&m_lep_isTrigMatched);
  
}

/// Reset all branch variables for leptons (called in execute)
void PhysLiteToOpenData::resetLeptonInfo(){

  m_ScaleFactor_ELE = 1.0;
  m_ScaleFactor_MUON = 1.0;
  m_ScaleFactor_LepTRIGGER = 1.0;
  m_ScaleFactor_ElTRIGGER = 1.0;
  m_ScaleFactor_MuTRIGGER = 1.0;

  m_lep_n = 0;
  m_lep_type->clear();
  m_lep_pt->clear();
  m_lep_eta->clear();
  m_lep_phi->clear();
  m_lep_e->clear();
  m_lep_charge->clear();
  m_lep_ptvarcone30->clear();
  m_lep_topoetcone20->clear();
  m_lep_z0->clear();
  m_lep_d0->clear();
  m_lep_d0sig->clear();
  m_lep_isTightID->clear();
  m_lep_isMediumID->clear();
  m_lep_isLooseID->clear();
  m_lep_isTightIso->clear();
  m_lep_isLooseIso->clear();
  m_lep_isTrigMatched->clear();
}

/// Fill event information into lepton variables and branches (called in execute)
StatusCode PhysLiteToOpenData::fillLeptonInfo(){

  /// Get the beamspot width in x, y, and xy for the event
  float beam_pos_sigma_x = m_event->eventInfo->beamPosSigmaX();
  float beam_pos_sigma_y = m_event->eventInfo->beamPosSigmaY();
  float beam_pos_sigma_xy = m_event->eventInfo->beamPosSigmaXY();

  /// Get the container of primary vertices for the event
  const xAOD::VertexContainer* vertices(nullptr);
  ANA_CHECK (evtStore()->retrieve(vertices, "PrimaryVertices"));

  /// Get the primary vertex z-position
  float vtx_z = 0;
  /// Loop over all the primary vertices
  for (const auto vtx : *vertices) {
    /// Include only primary vertices (e.g. no secondary vertices)
    if (vtx->vertexType() == xAOD::VxType::PriVtx) {
      /// Vertices in the container are ordered by sum pT^2, which is
      /// exactly what we want, so that the first one is the one we want
      vtx_z = vtx->z();
      break;
    }
  } /// Done with loop over primary vertices

  /// Loop over electrons to fill the lepton branches
  for (const xAOD::Electron* electron : *m_event->electrons) {
    /// Ensure that the electron passes the overlap-removal pre-selection, overlap-removal selection, and kinematic criteria
    if( !electron->auxdata<char>("preselectOR") ) continue;
    if( !electron->auxdata<char>("passesOR") ) continue;
    if( !electron->auxdata<unsigned int>("selectPtEta") ) continue;

    /// Store the lepton type: 11 for electrons
    m_lep_type->push_back(11);

    /// Store the electron kinematics (convert to GeV)
    m_lep_pt->push_back(electron->pt()*0.001);
    m_lep_eta->push_back(electron->eta());
    m_lep_phi->push_back(electron->phi());
    m_lep_e->push_back(electron->e()*0.001);

    /// Store the electron charge
    m_lep_charge->push_back(electron->charge());

    /// Store the electron isolation information (convert to GeV)
    m_lep_ptvarcone30->push_back(electron->isolation(xAOD::Iso::IsolationType::ptvarcone30_Nonprompt_All_MaxWeightTTVALooseCone_pt1000)*0.001);
    m_lep_topoetcone20->push_back(electron->isolation(xAOD::Iso::IsolationType::topoetcone20)*0.001);

    /// Calculate and store the z0 (displacement in z) with respect to the primary vertex for the electron
    float delta_z0 = electron->trackParticle()->z0() + electron->trackParticle()->vz() - vtx_z;
    m_lep_z0->push_back(delta_z0);

    /// Store the d0 (transverse impact parameter) with respect to the primary vertex for the electron
    m_lep_d0->push_back(electron->trackParticle()->d0());

    /// Calculate and store the d0 significance with respect to the primiary vertex for the electron
    /// Uses the beamspot information we got at the beginning of this method
    double d0sig = xAOD::TrackingHelpers::d0significance(electron->trackParticle(),
                                                         beam_pos_sigma_x,
                                                         beam_pos_sigma_y,
                                                         beam_pos_sigma_xy);
    m_lep_d0sig->push_back(d0sig);
    bool isTrigMatch = false;
    /// Get the loose, medium and tight ID and isolation decisions for the electron
    bool isTightID = bool(electron->auxdata< char >("DFCommonElectronsLHTight"));
    bool isMediumID = bool(electron->auxdata< char >("DFCommonElectronsLHMedium"));
    bool isLooseID = bool(electron->auxdata< char >("DFCommonElectronsLHLooseBL"));
    bool isTightIso = accEx_isolated_electron_tightiso->getBool(*electron);
    bool isLooseIso = accEx_isolated_electron_loose->getBool(*electron);

    /// Add three variables: passing tight ID, passing medium ID and passing loose ID
    m_lep_isTightID->push_back(isTightID);
    m_lep_isMediumID->push_back(isMediumID);
    m_lep_isLooseID->push_back(isLooseID);

    /// Add two variables: passing tight isolation, passing tight isolation
    m_lep_isTightIso->push_back(isTightIso);
    m_lep_isLooseIso->push_back(isLooseIso);
    
    
    // If event was triggered by the single-electron trigger check if this electron is the electron firing the trigger (i.e. matched to the trigger)
    if(m_trigE){
      // Data from 2015 or MC simulated event assigned to 2015 data taking
      if(m_randomRunNumber <= 284500){
        // The electron needs to have a transverse momentum which is larger than the trigger threshold
        // E.g. or HLT_e24_lhmedium_L1EM20VH the trigger threshold is 24 GeV and thus we require the electron
        // to have pT >= 25 GeV - similar for the other triggers
        if((electron->auxdata< char >("trigMatched_HLT_e24_lhmedium_L1EM20VH") && electron->pt() > 25000) ||
           (electron->isAvailable< char >("trigMatched_HLT_e60_lhmedium") && electron->auxdata< char >("trigMatched_HLT_e60_lhmedium") && electron->pt() > 61000) ||
           (electron->isAvailable< char >("trigMatched_HLT_e120_lhloose") && electron->auxdata< char >("trigMatched_HLT_e120_lhloose") && electron->pt() > 121000)){
          isTrigMatch = true;
          // If simulated events retrieve the trigger scale factors
          if( m_dataType != "data" ){
            m_ScaleFactor_LepTRIGGER *= electron->auxdata< float >("el_trigEffSF_e24_lhmedium_L1EM20VH_OR_e60_lhmedium_OR_e120_lhloose_NOSYS") ? abs(electron->auxdata< float >("el_trigEffSF_e24_lhmedium_L1EM20VH_OR_e60_lhmedium_OR_e120_lhloose_NOSYS")) : 1.0;
            m_ScaleFactor_ElTRIGGER *= electron->auxdata< float >("el_trigEffSF_e24_lhmedium_L1EM20VH_OR_e60_lhmedium_OR_e120_lhloose_NOSYS") ? abs(electron->auxdata< float >("el_trigEffSF_e24_lhmedium_L1EM20VH_OR_e60_lhmedium_OR_e120_lhloose_NOSYS")) : 1.0;
          }
        }
        // Data from 2016 or MC simulated event assigned to 2016 data taking
      }else if((m_randomRunNumber > 284500 && m_randomRunNumber < 320000)){
        // The electron needs to have a transverse momentum which is larger than the trigger threshold
        // E.g. or HLT_e26_lhtight_nod0_ivarloose the trigger threshold is 26 GeV and thus we require the electron
        // to have pT >= 27 GeV - similar for the other triggers
        //if(electron->isAvailable< char >("trigMatched_HLT_e26_lhtight_nod0_ivarloose") && electron->isAvailable< char >("trigMatched_HLT_e60_lhmedium_nod0") && electron->isAvailable< char >("trigMatched_HLT_e140_lhloose_nod0")){
        if((electron->auxdata< char >("trigMatched_HLT_e26_lhtight_nod0_ivarloose")  && electron->pt() > 27000) ||
           (electron->auxdata< char >("trigMatched_HLT_e60_lhmedium_nod0")  && electron->pt() > 61000) ||
           (electron->auxdata< char >("trigMatched_HLT_e140_lhloose_nod0")  && electron->pt() > 141000)){
          isTrigMatch = true;
          // If simulated events retrieve the trigger scale factors
          if( m_dataType != "data" ){
            m_ScaleFactor_LepTRIGGER *= electron->auxdata< float >("el_trigEffSF_e26_lhtight_nod0_ivarloose_OR_e60_lhmedium_nod0_OR_e140_lhloose_nod0_NOSYS") ? abs(electron->auxdata< float >("el_trigEffSF_e26_lhtight_nod0_ivarloose_OR_e60_lhmedium_nod0_OR_e140_lhloose_nod0_NOSYS")) : 1.0;
            m_ScaleFactor_ElTRIGGER *= electron->auxdata< float >("el_trigEffSF_e26_lhtight_nod0_ivarloose_OR_e60_lhmedium_nod0_OR_e140_lhloose_nod0_NOSYS") ? abs(electron->auxdata< float >("el_trigEffSF_e26_lhtight_nod0_ivarloose_OR_e60_lhmedium_nod0_OR_e140_lhloose_nod0_NOSYS")) : 1.0;
          }
          //std::cout<<"trigger eff electron "<<m_lep_n<<" = "<<electron->auxdata< float >("el_trigEffSF_e26_lhtight_nod0_ivarloose_OR_e60_lhmedium_nod0_OR_e140_lhloose_nod0_NOSYS")<<std::endl;
        }
        //}
      }
    }
    /**
    /// In the case of MC simulation, calculate the event-level lepton scale factor
    /// which is the product of the ID, reconstruction, and isolation scale factors
    /// for all selected electrons
    if( m_dataType != "data" ){
    Not yet supported for run2 rel > 22 (https://gitlab.cern.ch/atlas/athena/-/merge_requests/72590)
    m_ScaleFactor_ELE *= electron->auxdata< float >("el_reco_effSF_electron_loose_NOSYS");
    m_ScaleFactor_ELE *= electron->auxdata< float >("el_id_effSF_electron_loose_NOSYS");
    m_ScaleFactor_ELE *= electron->auxdata< float >("el_isol_effSF_electron_loose_NOSYS");
    }
    */
    m_lep_isTrigMatched->push_back(isTrigMatch);
    /// Increment the lepton counter
    m_lep_n++;
  } /// All done with the electron loop!

  /// Loop over mmuons to fill the lepton branches
  for (const xAOD::Muon* muon : *m_event->muons) {
    
    /// Ensure that the muon passes the overlap-removal pre-selection, overlap-removal selection, and kinematic criteria
    if( !muon->auxdata<char>("preselectOR") ) continue;
    if( !muon->auxdata<char>("passesOR") ) continue;
    if( !muon->auxdata<unsigned int>("selectPtEta") ) continue;

    /// Specifically in MC simulation, check that the reco/ID, isolation efficiency and TTVA scale factor calculation algorithm are set up correctly
    if( m_dataType != "data" ){
      /// check the reco/ID scale factor calculation algorithm
      if( muon->isAvailable< unsigned int >("muon_reco_bad_eff_medium") ){
        if( !bool(muon->auxdata< unsigned int >("muon_reco_bad_eff_medium")) ){ continue; }
      }
      else{ continue; }

      ///check the isolation efficiency scale factor calculation algorithm
      if( muon->isAvailable< unsigned int >("muon_isol_bad_eff_medium") ){
        if( !bool(muon->auxdata< unsigned int >("muon_isol_bad_eff_medium")) ){ continue; }
      }
      else{ continue; }
      /// check the TTVA scale factor calculation algorithm 
      if( muon->isAvailable< unsigned int >("muon_TTVA_bad_eff_medium") ){
        if( !bool(muon->auxdata< unsigned int >("muon_TTVA_bad_eff_medium")) ){ continue; }
      }
      else{ continue; }
    } 

    /// Store the lepton type: 13 for muons
    m_lep_type->push_back(13);

    /// Store the muon kinematics (convert to GeV)
    m_lep_pt->push_back(muon->pt()*0.001);
    m_lep_eta->push_back(muon->eta());
    m_lep_phi->push_back(muon->phi());
    m_lep_e->push_back(muon->e()*0.001);

    /// Store the muon charge
    m_lep_charge->push_back(muon->charge());

    /// Store the muon isolation information (convert to GeV)
    m_lep_ptvarcone30->push_back(muon->isolation(xAOD::Iso::IsolationType::ptvarcone30_Nonprompt_All_MaxWeightTTVA_pt1000)*0.001);
    m_lep_topoetcone20->push_back(muon->isolation(xAOD::Iso::IsolationType::topoetcone20)*0.001);

    /// Calculate and store the z0 (displacement in z) with respect to the primary vertex for the muon
    float delta_z0 = muon->primaryTrackParticle()->z0() + muon->primaryTrackParticle()->vz() - vtx_z;
    m_lep_z0->push_back(delta_z0);

    /// Store the d0 (transverse impact parameter) with respect to the primary vertex for the muon
    m_lep_d0->push_back(muon->primaryTrackParticle()->d0());

    /// Calculate and store the d0 significance with respect to the primiary vertex for the muon
    /// Uses the beamspot information we got at the beginning of this method
    double d0sig = xAOD::TrackingHelpers::d0significance(muon->primaryTrackParticle(),
                                                         beam_pos_sigma_x,
                                                         beam_pos_sigma_y,
                                                         beam_pos_sigma_xy);
    m_lep_d0sig->push_back(d0sig);

    bool isTrigMatch = false;
    /// Get the loose, medium and tight ID and isolation decisions for the muon  
    bool isTightID = accEx_good_muon_tight->getBool(*muon);
    bool isMediumID = accEx_good_muon_medium->getBool(*muon);
    bool isLooseID = accEx_good_muon_loose->getBool(*muon);
    bool isTightIso = accEx_isolated_muon_tightiso->getBool(*muon);
    bool isLooseIso = accEx_isolated_muon_medium->getBool(*muon);

    /// Add three variables: passing tight ID, passing medium ID, and passing loose ID
    m_lep_isTightID->push_back(isTightID);
    m_lep_isMediumID->push_back(isMediumID);
    m_lep_isLooseID->push_back(isLooseID);

    /// Add two variables: passing tight isolation and passing loose isolation
    m_lep_isTightIso->push_back(isTightIso);
    m_lep_isLooseIso->push_back(isLooseIso);
    
   // If event was triggered by the single-muon trigger check if this electron is the muon firing the trigger (i.e. matched to the trigger)
    if(m_trigM){
      // Data from 2015 or MC simulated event assigned to 2015 data taking
      if(m_randomRunNumber <= 284500){
        // The muon needs to have a transverse momentum which is larger than the trigger threshold
        // E.g. or HLT_mu20_iloose_L1MU15 the trigger threshold is 20 GeV and thus we require the muon
        // to have pT >= 21 GeV - similar for the other triggers
        //if( muon->isAvailable< char >("trigMatched_HLT_mu20_iloose_L1MU15") && muon->isAvailable< char >("trigMatched_HLT_mu40")){
        if((muon->auxdata< char >("trigMatched_HLT_mu20_iloose_L1MU15") && muon->pt() > 21000) ||
           (muon->auxdata< char >("trigMatched_HLT_mu40") && muon->pt() > 41000)){
          isTrigMatch = true;
          // If simulated events retrieve the trigger scale factors
          if( m_dataType != "data" ){
            m_ScaleFactor_LepTRIGGER *= muon->auxdata< float >("muon_trigEffSF_mu20_iloose_L1MU15_OR_mu40_NOSYS") ? abs(muon->auxdata< float >("muon_trigEffSF_mu20_iloose_L1MU15_OR_mu40_NOSYS")) : 1.0;
            m_ScaleFactor_MuTRIGGER *= muon->auxdata< float >("muon_trigEffSF_mu20_iloose_L1MU15_OR_mu40_NOSYS") ? abs(muon->auxdata< float >("muon_trigEffSF_mu20_iloose_L1MU15_OR_mu40_NOSYS")) : 1.0;
          }
        }
        //}
      }
      // Data from 2016 or MC simulated event assigned to 2016 data taking
      if((m_randomRunNumber > 284500 && m_randomRunNumber < 320000)){
        // The muon needs to have a transverse momentum which is larger than the trigger threshold
        // E.g. or HLT_mu26_ivarmedium the trigger threshold is 26 GeV and thus we require the muon
        // to have pT >= 27 GeV - similar for the other triggers
        //if( muon->isAvailable< char >("trigMatched_HLT_mu26_ivarmedium") && muon->isAvailable< char >("trigMatched_HLT_mu50")){  
        if((muon->auxdata< char >("trigMatched_HLT_mu26_ivarmedium") && muon->pt() > 27000) ||
           (muon->auxdata< char >("trigMatched_HLT_mu50")  && muon->pt() > 51000)){
          isTrigMatch = true;
          // If simulated events retrieve the trigger scale factors
          if( m_dataType != "data" ){
            m_ScaleFactor_LepTRIGGER *= muon->auxdata< float >("muon_trigEffSF_mu26_ivarmedium_OR_mu50_NOSYS") ? abs(muon->auxdata< float >("muon_trigEffSF_mu26_ivarmedium_OR_mu50_NOSYS")) : 1.0;
            m_ScaleFactor_MuTRIGGER *= muon->auxdata< float >("muon_trigEffSF_mu26_ivarmedium_OR_mu50_NOSYS") ? abs(muon->auxdata< float >("muon_trigEffSF_mu26_ivarmedium_OR_mu50_NOSYS")) : 1.0;
          }
        }
        //}
      }
    }
    /// In the case of MC simulation, calculate the event-level lepton scale factor
    /// which is the product of the ID/reconstruction, isolation, and track-to-vertex
    /// association scale factors for all selected muons
    if( m_dataType != "data" ){
      m_ScaleFactor_MUON *= muon->auxdata< float >("muon_reco_effSF_medium_NOSYS");
      m_ScaleFactor_MUON *= muon->auxdata< float >("muon_isol_effSF_medium_NOSYS");
      m_ScaleFactor_MUON *= muon->auxdata< float >("muon_TTVA_effSF_medium_NOSYS");
    }
    
    m_lep_isTrigMatched->push_back(isTrigMatch);
    /// Increment the lepton counter
    m_lep_n++;
  } /// All done with the muon loop!
  
  return StatusCode::SUCCESS;
}

/// Cleanup function - delete all our vector branches (called in destructor)
void PhysLiteToOpenData::cleanupLeptonInfo(){

  delete m_lep_type;
  delete m_lep_pt;
  delete m_lep_eta;
  delete m_lep_phi;
  delete m_lep_e;
  delete m_lep_charge;
  delete m_lep_ptvarcone30;
  delete m_lep_topoetcone20;
  delete m_lep_z0;
  delete m_lep_d0;
  delete m_lep_d0sig;
  delete m_lep_isTightID;
  delete m_lep_isMediumID;
  delete m_lep_isLooseID;
  delete m_lep_isTightIso;
  delete m_lep_isLooseIso;
  delete m_lep_isTrigMatched;

}
