/// Header for the class
#include <PhysLiteToOpenData/PhysLiteToOpenData.h>

/// Initialize all our photon branches and variables (called in initialize)
void PhysLiteToOpenData::initPhotonInfo(TTree * mytree){

  // Event-level photon scale factor
  m_ScaleFactor_PHOTON = 1.0;
  mytree->Branch("ScaleFactor_PHOTON",&m_ScaleFactor_PHOTON);

  // Photon variables (see header for details)
  m_photon_n = 0;
  m_photon_pt = new std::vector<float>();
  m_photon_eta = new std::vector<float>();
  m_photon_phi = new std::vector<float>();
  m_photon_e = new std::vector<float>();
  m_photon_ptcone20 = new std::vector<float>();
  m_photon_topoetcone40 = new std::vector<float>();
  m_photon_isLooseID = new std::vector<bool>();
  m_photon_isTightID = new std::vector<bool>();
  m_photon_isLooseIso = new std::vector<bool>();
  m_photon_isTightIso = new std::vector<bool>();
  
  mytree->Branch("photon_n",&m_photon_n);
  mytree->Branch("photon_pt",&m_photon_pt);
  mytree->Branch("photon_eta",&m_photon_eta);
  mytree->Branch("photon_phi",&m_photon_phi);
  mytree->Branch("photon_e",&m_photon_e);
  mytree->Branch("photon_ptcone20",&m_photon_ptcone20);
  mytree->Branch("photon_topoetcone40",&m_photon_topoetcone40);
  mytree->Branch("photon_isLooseID",&m_photon_isLooseID);
  mytree->Branch("photon_isTightID",&m_photon_isTightID);
  mytree->Branch("photon_isLooseIso",&m_photon_isLooseIso);
  mytree->Branch("photon_isTightIso",&m_photon_isTightIso);

}

/// Reset all branch variables for photons (called in execute)
void PhysLiteToOpenData::resetPhotonInfo(){

  m_ScaleFactor_PHOTON = 1.0;

  m_photon_n = 0;
  m_photon_pt->clear();
  m_photon_eta->clear();
  m_photon_phi->clear();
  m_photon_e->clear();
  m_photon_ptcone20->clear();
  m_photon_topoetcone40->clear();
  m_photon_isLooseID->clear();
  m_photon_isTightID->clear();
  m_photon_isLooseIso->clear();
  m_photon_isTightIso->clear();

}

/// Fill event information into photon variables and branches (called in execute)
StatusCode PhysLiteToOpenData::fillPhotonInfo(){
  
  for (const xAOD::Photon* photon : *m_event->photons ) {
    /// Ensure that the photon passes the overlap-removal pre-selection, overlap-removal selection, and kinematic criteria
    if( !photon->auxdata<char>("preselectOR") ) continue;
    if( !photon->auxdata<char>("passesOR") ) continue;
    if( !photon->auxdata<unsigned int>("selectPtEta") ) continue;


    /// Store the photon kinematics (covert to GeV)
    m_photon_pt->push_back(photon->pt()*0.001);
    m_photon_eta->push_back(photon->eta());
    m_photon_phi->push_back(photon->phi());
    m_photon_e->push_back(photon->e()*0.001);

    /// Store the photon isolation variables (convert to GeV)
    m_photon_ptcone20->push_back(photon->isolation(xAOD::Iso::IsolationType::ptcone20)*0.001);
    m_photon_topoetcone40->push_back(photon->isolation(xAOD::Iso::IsolationType::topoetcone40)*0.001);
    /// Get the loose and tight ID and isolation decisions for the photon
    bool isTightID  = accEx_selectEM_photon_tight->getBool(*photon);
    bool isLooseID  = accEx_selectEM_photon_loose->getBool(*photon);
    bool isLooseIso = accEx_isolated_photon_loose->getBool(*photon);
    bool isTightIso = accEx_isolated_photon_tightiso->getBool(*photon);

    /// Add four variables: passing tight ID, passing loose ID, passing loose isolation and passing tight isolation
    m_photon_isTightID->push_back(isTightID);
    m_photon_isLooseID->push_back(isLooseID);
    m_photon_isLooseIso->push_back(isLooseIso);
    m_photon_isTightIso->push_back(isTightIso);

    /// In the case of MC simulation, calculate the event-level photon scale factor
    /// which is the product of the ID and isolation scale factors for all selected photons
    if( m_dataType != "data" ){
      m_ScaleFactor_PHOTON *= photon->auxdata< float >("ph_id_effSF_photon_loose_NOSYS");
      m_ScaleFactor_PHOTON *= photon->auxdata< float >("ph_isol_effSF_photon_loose_NOSYS");
    }

    /// Increment the photon counter
    m_photon_n++;
  } /// All done with the photon loop!

  return StatusCode::SUCCESS;
}

/// Cleanup function - delete all our vector branches (called in destructor)
void PhysLiteToOpenData::cleanupPhotonInfo(){

  delete m_photon_pt;
  delete m_photon_eta;
  delete m_photon_phi;
  delete m_photon_e;
  delete m_photon_ptcone20;
  delete m_photon_topoetcone40;
  delete m_photon_isLooseID;
  delete m_photon_isTightID;
  delete m_photon_isLooseIso;
  delete m_photon_isTightIso;

}
