/// Header for this class
#include <PhysLiteToOpenData/PhysLiteToOpenData.h>

/// Concrete implementation of PMG cross-section tool
#include <PMGTools/PMGCrossSectionTool.h>

/// Concrete implementation of the jet energy scale uncertainties tool
#include <JetUncertainties/JetUncertaintiesTool.h>

/// Systematic set for applying systematic variations
#include <PATInterfaces/SystematicSet.h>

/// Special correction code return value for applying calibrations
#include <PATInterfaces/CorrectionCode.h>

/// Path resolver for finding files
#include <PathResolver/PathResolver.h>


/// Constructor; nothing to do but initialize the base class
PhysLiteToOpenData :: PhysLiteToOpenData (const std::string& name,
                                  ISvcLocator *pSvcLocator)
  : EL::AnaAlgorithm (name, pSvcLocator)
{
  // Here you put any code for the base initialization of variables,
  // e.g. initialize all pointers to 0.  This is also where you
  // declare all properties for your algorithm.  Note that things like
  // resetting statistics variables or booking histograms should
  // rather go into the initialize() function.
}


/// Initialize - called once per job
StatusCode PhysLiteToOpenData :: initialize ()
{
  // Here you do everything that needs to be done at the very
  // beginning on each worker node, e.g. create histograms and output
  // trees.  This method gets called before any input files are
  // connected.
  ANA_MSG_INFO ("In initialize");

  // Set up the PMG cross section tool and initialize it using a file on cvmfs
  ASG_SET_ANA_TOOL_TYPE( m_PMGCrossSectionTool, PMGTools::PMGCrossSectionTool);
  m_PMGCrossSectionTool.setName("myCrossSectionTool");
  ANA_CHECK(m_PMGCrossSectionTool.retrieve());
  m_PMGCrossSectionTool->readInfosFromFiles({PathResolverFindCalibFile("PhysLiteToOpenData/PMGxsecDB_mc16.txt")});

  // Set up the tool for jet energy scale uncertainties as well
  ASG_SET_ANA_TOOL_TYPE( m_jetUncertaintiesTool, JetUncertaintiesTool);
  m_jetUncertaintiesTool.setName("PhysLiteToOpenData_JESProvider");
  ANA_CHECK(m_jetUncertaintiesTool.setProperty("JetDefinition","AntiKt4EMPFlow"));
  ANA_CHECK(m_jetUncertaintiesTool.setProperty("MCType","MC20"));
  ANA_CHECK(m_jetUncertaintiesTool.setProperty("ConfigFile","rel22/Summer2023_PreRec/R4_SR_Scenario1_SimpleJER.config"));
  ANA_CHECK(m_jetUncertaintiesTool.setProperty("IsData",m_dataType=="data"));
  ANA_CHECK(m_jetUncertaintiesTool.initialize());

  // Initialize the output tree, using the standard name, "analysis"
  ANA_CHECK (book (TTree ("analysis", "My analysis ntuple")));
  TTree* mytree = tree ("analysis");

  // Set up the accessors for selection criteria
  ANA_CHECK( makeSelectionReadAccessor("isolated_electron_loose,as_char", accEx_isolated_electron_loose) );
  ANA_CHECK( makeSelectionReadAccessor("isolated_electron_tightiso,as_char", accEx_isolated_electron_tightiso) );
  ANA_CHECK( makeSelectionReadAccessor("selectLikelihood_electron_loose,as_char", accEx_selectLikelihood_electron_loose) );
  
  ANA_CHECK( makeSelectionReadAccessor("good_muon_loose,as_char", accEx_good_muon_loose) );
  ANA_CHECK( makeSelectionReadAccessor("good_muon_medium,as_char", accEx_good_muon_medium) );
  ANA_CHECK( makeSelectionReadAccessor("good_muon_tight,as_char", accEx_good_muon_tight) );
  
  ANA_CHECK( makeSelectionReadAccessor("isolated_muon_medium,as_char", accEx_isolated_muon_medium) );
  ANA_CHECK( makeSelectionReadAccessor("isolated_muon_tightiso,as_char", accEx_isolated_muon_tightiso) );
  
  ANA_CHECK( makeSelectionReadAccessor("isolated_photon_loose,as_char", accEx_isolated_photon_loose) );
  ANA_CHECK( makeSelectionReadAccessor("isolated_photon_tightiso,as_char", accEx_isolated_photon_tightiso) );
  ANA_CHECK( makeSelectionReadAccessor("selectEM_photon_loose,as_char", accEx_selectEM_photon_loose) );
  ANA_CHECK( makeSelectionReadAccessor("selectEM_photon_medium,as_char", accEx_selectEM_photon_medium) );
  ANA_CHECK( makeSelectionReadAccessor("selectEM_photon_tight,as_char", accEx_selectEM_photon_tight) );
  
  ANA_CHECK( makeSelectionReadAccessor("selected_tau_tight,as_char", accEx_selected_tau_tight) );

  // Initialize the event object
  m_event = new Event();

  // Initialize the event-level weight variables and add them to the output tree
  m_ScaleFactor_PILEUP = 1.0;
  m_TriggerMatch_Dilepton = false;
  m_ScaleFactor_MLTRIGGER = 1.0;
  m_ScaleFactor_FTAG = 1.0;
  m_mcWeight = 1.0;
  mytree->Branch("TriggerMatch_DILEPTON",&m_TriggerMatch_Dilepton);
  mytree->Branch("ScaleFactor_MLTRIGGER",&m_ScaleFactor_MLTRIGGER);
  mytree->Branch("ScaleFactor_PILEUP",&m_ScaleFactor_PILEUP);
  mytree->Branch("ScaleFactor_FTAG",&m_ScaleFactor_FTAG);
  mytree->Branch("mcWeight",&m_mcWeight);

  m_channelNumber = 0; 
  // Initialize the cross-section, filter efficiency, and k-factor branches and add them to the output tree
  m_xsec = m_filteff = m_kfac = 0;
  mytree->Branch("xsec",&m_xsec);
  mytree->Branch("filteff",&m_filteff);
  mytree->Branch("kfac",&m_kfac);

  mytree->Branch("channelNumber",&m_channelNumber,"channelNumber/i");
  mytree->Branch("eventNumber",&m_eventNumber,"eventNumber/l");
  mytree->Branch("runNumber",&m_runNumber,"runNumber/i");

  // Initialize the trigger variables and add them to the tree
  m_trigP = false;
  m_trigDT = false;
  m_trigT = false;
  m_trigE = false;
  m_trigDE = false;
  m_trigM = false;
  m_trigDM = false;
  m_trigMET = false;
  m_trigML = false;

  mytree->Branch("trigML",&m_trigML);  
  mytree->Branch("trigP",&m_trigP);
  mytree->Branch("trigDT",&m_trigDT);
  mytree->Branch("trigT",&m_trigT);
  mytree->Branch("trigE",&m_trigE);
  mytree->Branch("trigDM",&m_trigDM);
  mytree->Branch("trigDE",&m_trigDE);
  mytree->Branch("trigM",&m_trigM);
  mytree->Branch("trigMET",&m_trigMET);

  // Call the initialization helper functions for the physics object branches / variables
  initJetInfo(mytree);
  initLeptonInfo(mytree);
  initPhotonInfo(mytree);
  initTauInfo(mytree);
  // Altough truth information is only for MC we add empty variables to data
  initTruthInfo(mytree);

  // Initialize the MET variables and add them to the output tree
  m_met = 0;
  m_met_phi = 0;
  m_met_mpx = 0;
  m_met_mpy = 0;
  mytree->Branch ("met", &m_met);
  mytree->Branch ("met_phi", &m_met_phi);
  mytree->Branch ("met_mpx", &m_met_mpx);
  mytree->Branch ("met_mpy", &m_met_mpy);

  // All done, no problems!
  return StatusCode::SUCCESS;
}

StatusCode PhysLiteToOpenData :: fillEvent ()
{
  // Populate the Event object with all the collections we want for this event
  ANA_CHECK (evtStore()->retrieve (m_event->eventInfo, "EventInfo"));

  // All physics ojbects are "NOSYS": nominal calibrations are applied
  ANA_CHECK (evtStore()->retrieve (m_event->jets, "AnalysisJets_NOSYS"));
  ANA_CHECK (evtStore()->retrieve (m_event->electrons, "AnalysisElectrons_NOSYS"));
  ANA_CHECK (evtStore()->retrieve (m_event->muons, "AnalysisMuons_NOSYS"));
  ANA_CHECK (evtStore()->retrieve (m_event->taus, "AnalysisTauJets_NOSYS"));
  ANA_CHECK (evtStore()->retrieve (m_event->photons, "AnalysisPhotons_NOSYS"));
  ANA_CHECK (evtStore()->retrieve (m_event->allMET, "AnalysisMET_NOSYS"));
  ANA_CHECK (evtStore()->retrieve (m_event->largeRJets, "AnalysisLargeRJets_NOSYS"));

  // Truth information only for MC
  if( m_dataType != "data" ){
    ANA_CHECK (evtStore()->retrieve (m_event->truthJets, "AntiKt4TruthDressedWZJets"));
    ANA_CHECK (evtStore()->retrieve (m_event->truthElec, "TruthElectrons"));
    ANA_CHECK (evtStore()->retrieve (m_event->truthMuon, "TruthMuons"));
    ANA_CHECK (evtStore()->retrieve (m_event->truthTaus, "TruthTaus"));
    ANA_CHECK (evtStore()->retrieve (m_event->truthPhot, "TruthPhotons"));
    ANA_CHECK (evtStore()->retrieve (m_event->truthMET, "MET_Truth"));
  }

  return StatusCode::SUCCESS;
}

StatusCode PhysLiteToOpenData :: execute ()
{
  // Here you do everything that needs to be done on every single
  // events, e.g. read input variables, apply cuts, and fill
  // histograms and trees.  This is where most of your actual analysis
  // code will go.

  // Initialize all our simple float / weight variables
  m_ScaleFactor_PILEUP = 1.0;
  m_ScaleFactor_MLTRIGGER = 1.0;
  m_ScaleFactor_FTAG = 1.0;
  m_mcWeight = 1.0;

  m_trigE = m_trigM = m_trigMET = m_trigP = m_trigDT = m_trigT = m_trigDE = m_trigDM = false;

  m_TriggerMatch_Dilepton = false;
  
  //*********************************************************
  // Init value for cross section taken with the PMG tool
  //*********************************************************
  m_channelNumber = 0;
  m_eventNumber = 0;
  m_runNumber = 0;
  m_xsec = 0;
  m_randomRunNumber = 0;

  // Reset the branches for all the physics objects
  resetJetInfo();
  resetLeptonInfo();
  resetPhotonInfo();
  resetTauInfo();
  // Truth is only available for simulations, but since we added the (empty) variables also for data we can reset them here (altough they are not filled)
  resetTruthInfo();
  // Reset the MET variables
  m_met = 0;
  m_met_phi = 0;
  m_met_mpx = 0;
  m_met_mpy = 0;

  // Fill the event information into our Event class
  ANA_CHECK (fillEvent());

  m_runNumber = m_event->eventInfo->runNumber();
  // If we are working with MC simulation, we need additional information
  if( m_dataType != "data" ){
    // Get the pileup weight from the event info
    m_ScaleFactor_PILEUP = m_event->eventInfo->auxdata< float >("PileupWeight_NOSYS");
    // Get the MC event weight (generator weight) from the event info
    m_mcWeight = m_event->eventInfo->auxdata< float >("generatorWeight_NOSYS");
    // Get the dataset identifier (DSID, unique integer for each dataset) from the event info
    m_channelNumber = m_event->eventInfo->auxdata<unsigned int>("mcChannelNumber");
    m_eventNumber = m_event->eventInfo->mcEventNumber();
    // Get the cross section (xsec), filter efficiency, and k-factor from the PMG tool
    m_xsec = m_PMGCrossSectionTool->getAMIXsection(m_channelNumber);
    m_filteff = m_PMGCrossSectionTool->getFilterEff(m_channelNumber);
    m_kfac = m_PMGCrossSectionTool->getKfactor(m_channelNumber);
    // Needed for trigger matching and trigger scale factors
    m_randomRunNumber = m_event->eventInfo->auxdata<unsigned int>("RandomRunNumber");
    m_ScaleFactor_MLTRIGGER = m_event->eventInfo->auxdata< float >("globalTriggerEffSF_NOSYS");
    // The scale factor for flavour tagging (aka. b-tagging)
    m_ScaleFactor_FTAG = m_event->eventInfo->auxdata< float >("ftag_effSF_DL1dv01_Continuous_NOSYS");
  }else{
    // If this is data, take the metadata directly from the EventInfo
    m_channelNumber = m_event->eventInfo->runNumber();
    m_eventNumber = m_event->eventInfo->eventNumber();
    /// The RandomRunNumber is only used in MC to assign the simulated events to
    // either 2015 or 2016 data taking. For data it is filled with the actual run number.
    m_randomRunNumber = m_runNumber; 
  }

  // Check if any of the leptons in the event were matched to any of the di-lepton triggers 
  if(m_event->eventInfo->auxdata< char >("globalTriggerMatch_NOSYS"))m_TriggerMatch_Dilepton = true;

  // Check all our trigger decisions, and set the event-level trigger decision variables accordingly
  // If this is either data from 2015 or an event in the MC simulations belonging to the 2015 data taking; check 2015 trigger chains 
  if(m_randomRunNumber <= 284500){
    // Di-electron trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_2e12_lhloose_L12EM10VH") ) m_trigDE = true;
    // Di-muon trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_2mu10") ) m_trigDM = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_mu18_mu8noL1") ) m_trigDM = true;
    // Single-electron trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_e24_lhmedium_L1EM20VH") ) m_trigE = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_e60_lhmedium") ) m_trigE = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_e120_lhloose") ) m_trigE = true;
    // Single-muon trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_mu20_iloose_L1MU15") ) m_trigM = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_mu40") ) m_trigM = true;
    // MET trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_xe70_mht") ) m_trigMET = true;
    // Photon trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_g120_loose") ) m_trigP = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_g200_etcut") ) m_trigP = true;
    // Tau trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_tau80_medium1_tracktwo_L1TAU60") ) m_trigT = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_tau35_medium1_tracktwo_tau25_medium1_tracktwo") ) m_trigDT = true;
    // If this is either data from 2016 or an event in the MC simulations belonging to the 2016 data taking; check 2016 trigger chains     
  }else if(m_randomRunNumber > 284500 && m_randomRunNumber < 320000){
    // Di-electron trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_2e17_lhvloose_nod0") ) m_trigDE = true;
    // Di-muon trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_2mu14") ) m_trigDM = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_mu22_mu8noL1") ) m_trigDM = true;
    // Single-electron trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_e26_lhtight_nod0_ivarloose") ) m_trigE = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_e60_lhmedium_nod0") ) m_trigE = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_e140_lhloose_nod0") ) m_trigE = true;
    // Single-muon trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_mu26_ivarmedium") ) m_trigM = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_mu50") ) m_trigM = true;
    // MET trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_xe90_mht_L1XE50") ) m_trigMET = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_xe110_mht_L1XE50") ) m_trigMET = true;
    // Photon trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_g140_loose") ) m_trigP = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_g300_etcut") ) m_trigP = true;
    // Single/di-tau trigger chains
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_tau160_medium1_tracktwo") ) m_trigT = true;
    if( m_event->eventInfo->auxdata< bool >("trigPassed_HLT_tau35_medium1_tracktwo_tau25_medium1_tracktwo_L1TAU20IM_2TAU12IM") ) m_trigDT = true;

  }else{
    std::cout<<"ERROR \t Runnumber "<<m_randomRunNumber<<" does not belong to any period"<<std::endl;
    return StatusCode::FAILURE;
  }
  
  ANA_CHECK( fillJetInfo() );
  ANA_CHECK( fillLeptonInfo() );
  ANA_CHECK( fillPhotonInfo() );
  ANA_CHECK( fillTauInfo() );
  // Truth information (only filled for MC). For data empty variables are added.
  if( m_dataType != "data" ){
    ANA_CHECK( fillTruthInfo() );
  }

  // Get the final MET and store the scalar and phi direction - Convert to GeV as we go
  const xAOD::MissingET* met = (*m_event->allMET)["Final"];
  m_met = met->met()*0.001;
  m_met_phi = met->phi();
  m_met_mpx = met->mpx()*0.001;
  m_met_mpy = met->mpy()*0.001; 

  // Fill the event into the tree
  tree("analysis")->Fill();

  return StatusCode::SUCCESS;
}

// Finalize - if we want to output statistics or anything like that
// those would go here.
StatusCode PhysLiteToOpenData :: finalize ()
{
  // This method is the mirror image of initialize(), meaning it gets
  // called after the last event has been processed on the worker node
  // and allows you to finish up any objects you created in
  // initialize() before they are written to disk.  This is actually
  // fairly rare, since this happens separately for each worker node.
  // Most of the time you want to do your post-processing on the
  // submission node after all your histogram outputs have been
  // merged.
  return StatusCode::SUCCESS;
}

// Standard destructor
PhysLiteToOpenData :: ~PhysLiteToOpenData () {

  // Just clean up all the object-level variables (those are the vectors we're dealing with)
  cleanupJetInfo();
  cleanupLeptonInfo();
  cleanupPhotonInfo();
  cleanupTauInfo();
  // Truth information (only filled for MC)
  cleanupTruthInfo();
}
