/// Header for the class
#include <PhysLiteToOpenData/PhysLiteToOpenData.h>

/// Initialize all our tau jet branches and variables (called in initialize)
void PhysLiteToOpenData::initTauInfo(TTree * mytree){

  // Event-level tau scale factor
  m_ScaleFactor_TAU = 1.0;
  m_ScaleFactor_TauTRIGGER = 1.0;
  m_ScaleFactor_DiTauTRIGGER = 1.0;

  mytree->Branch("ScaleFactor_TAU",&m_ScaleFactor_TAU);
  mytree->Branch("ScaleFactor_TauTRIGGER",&m_ScaleFactor_TauTRIGGER);
  mytree->Branch("ScaleFactor_DiTauTRIGGER",&m_ScaleFactor_DiTauTRIGGER);

  // Tau-jet variables (see header for details)
  m_tau_n = 0;
  m_tau_pt = new std::vector<float>();
  m_tau_eta = new std::vector<float>();
  m_tau_phi = new std::vector<float>();
  m_tau_e = new std::vector<float>();
  m_tau_charge = new std::vector<float>();
  m_tau_nTracks = new std::vector<int>();
  m_tau_isTight = new std::vector<bool>();
  m_tau_RNNJetScore = new std::vector<float>();
  m_tau_RNNEleScore = new std::vector<float>();

  mytree->Branch("tau_n",&m_tau_n);
  mytree->Branch("tau_pt",&m_tau_pt);
  mytree->Branch("tau_eta",&m_tau_eta);
  mytree->Branch("tau_phi",&m_tau_phi);
  mytree->Branch("tau_e",&m_tau_e);
  mytree->Branch("tau_charge",&m_tau_charge);
  mytree->Branch("tau_nTracks",&m_tau_nTracks);
  mytree->Branch("tau_isTight",&m_tau_isTight);
  mytree->Branch("tau_RNNJetScore",&m_tau_RNNJetScore);
  mytree->Branch("tau_RNNEleScore",&m_tau_RNNEleScore);

}

/// Reset all branch variables for tau jets (called in execute)
void PhysLiteToOpenData::resetTauInfo(){

  m_ScaleFactor_TAU = 1.0;
  m_ScaleFactor_TauTRIGGER = 1.0;
  m_ScaleFactor_DiTauTRIGGER = 1.0;

  m_tau_n = 0;
  m_tau_pt->clear();
  m_tau_eta->clear();
  m_tau_phi->clear();
  m_tau_e->clear();
  m_tau_charge->clear();
  m_tau_nTracks->clear();
  m_tau_isTight->clear();
  m_tau_RNNJetScore->clear();
  m_tau_RNNEleScore->clear();

}

/// Fill event information into tau jet variables and branches (called in execute)
StatusCode PhysLiteToOpenData::fillTauInfo(){

  for (const xAOD::TauJet* tau : *m_event->taus ) {
    /// Ensure that the tau jet passes the overlap-removal pre-selection, overlap-removal selection, and kinematic criteria
    if( !tau->auxdata<char>("preselectOR") ) continue;
    if( !tau->auxdata<char>("passesOR") ) continue;
    if( !tau->auxdata<unsigned>("selectPtEta") ) continue;

    /// Store the tau jet kinematics (convert to GeV)
    m_tau_pt->push_back(tau->pt()*0.001);
    m_tau_eta->push_back(tau->eta());
    m_tau_phi->push_back(tau->phi());
    m_tau_e->push_back(tau->e()*0.001);

    /// Store the tau charge and number of charged tracks
    m_tau_charge->push_back(tau->charge());
    m_tau_nTracks->push_back(tau->nTracks());

    /// Did the tau pass our selection criteria? Use the accessor
    m_tau_isTight->push_back(accEx_selected_tau_tight->getBool(*tau));

    /// Store the RNN scores for the two RNNs used for tau ID
    m_tau_RNNJetScore->push_back(tau->auxdata< float >("RNNJetScore"));
    m_tau_RNNEleScore->push_back(tau->auxdata< float >("RNNEleScore"));

    /// In the case of MC simulation, calculate the event-level tau scale factor
    /// which is simply the product of the individual tau scale-factors
    if( m_dataType != "data" ){
      // If single-tau trigger fired the event, get the scale factor for these triggers
      if(m_trigT){
        m_ScaleFactor_TauTRIGGER *= tau->auxdata< float >("tau_trigEffSF_tau80L1TAU60_medium1_tracktwo_NOSYS") ? abs(tau->auxdata< float >("tau_trigEffSF_tau80L1TAU60_medium1_tracktwo_NOSYS")) : 1.0;
        m_ScaleFactor_TauTRIGGER *= tau->auxdata< float >("tau_trigEffSF_tau160_medium1_tracktwo_NOSYS") ? abs(tau->auxdata< float >("tau_trigEffSF_tau160_medium1_tracktwo_NOSYS")) : 1.0;
      }
      // If di-tau trigger fired the event, get the scale factor for these triggers
      if(m_trigDT){
        m_ScaleFactor_DiTauTRIGGER *= tau->auxdata< float >("tau_trigEffSF_tau35_medium1_tracktwo_NOSYS");
        m_ScaleFactor_DiTauTRIGGER *= tau->auxdata< float >("tau_trigEffSF_tau25_medium1_tracktwo_NOSYS");
      }
      // Adding the tau reconstruction and identification scale factors to the total event weight for taus
      m_ScaleFactor_TAU *= tau->auxdata< float >("tau_Reco_effSF_tight_NOSYS");
      m_ScaleFactor_TAU *= tau->auxdata< float >("tau_ID_effSF_tight_NOSYS");
    }

    /// Increment the tau counter
    m_tau_n++;
  } /// All done with the tau loop!
  
  return StatusCode::SUCCESS;
}

/// Cleanup function - delete all our vector branches (called in destructor)
void PhysLiteToOpenData::cleanupTauInfo(){

  delete m_tau_pt;
  delete m_tau_eta;
  delete m_tau_phi;
  delete m_tau_e;
  delete m_tau_charge;
  delete m_tau_nTracks;
  delete m_tau_isTight;
  delete m_tau_RNNJetScore;
  delete m_tau_RNNEleScore;

}
