# Configuration for the physics object calibration and uncertainty tools
from AnaAlgorithm.AlgSequence import AlgSequence
from AnaAlgorithm.DualUseConfig import createService, createAlgorithm, addPrivateTool
from AnalysisAlgorithmsConfig.ConfigSequence import ConfigSequence
from AnalysisAlgorithmsConfig.ConfigAccumulator import ConfigAccumulator, DataType
from AthenaConfiguration.Enums import LHCPeriod
from Campaigns.Utils import Campaign

# Add some cuts for leptons/photons/jets to select for the ntuples
jetMinPt = 20e3 # Units MeV
jetMaxEta = 2.5
muonMinPt = 7e3 # Units MeV
muonMaxEta = 2.5
electronMinPt = 7e3 # Units MeV
electronMaxEta = 2.47
photonMinPt = 25e3 # Units MeV
photonMaxEta = 2.37
tauMinPt = 20e3 # Units MeV
tauMaxEta = 2.5
largeRJetMinPt = 250e3 # Units MeV
largeRJetMaxEta = 2.0

# List of trigger chains;
triggerChains = [
    #2015
    'HLT_2e12_lhloose_L12EM10VH',
    'HLT_2mu10',
    'HLT_mu18_mu8noL1',
    "HLT_e24_lhmedium_L1EM20VH",
    "HLT_e60_lhmedium",
    "HLT_e120_lhloose",
    "HLT_mu20_iloose_L1MU15",
    "HLT_mu40",
    "HLT_xe70_mht",
    "HLT_g120_loose",
    "HLT_g200_etcut",
    "HLT_tau80_medium1_tracktwo_L1TAU60",
    "HLT_tau35_medium1_tracktwo_tau25_medium1_tracktwo",
    #2016
    'HLT_2e17_lhvloose_nod0',
    'HLT_2mu14',
    'HLT_mu22_mu8noL1', 
    "HLT_e26_lhtight_nod0_ivarloose",
    "HLT_e60_lhmedium_nod0",
    "HLT_e140_lhloose_nod0",
    "HLT_mu26_ivarmedium",
    "HLT_mu50",
    "HLT_xe90_mht_L1XE50",
    "HLT_xe110_mht_L1XE50",
    "HLT_g140_loose",
    "HLT_g300_etcut",
    "HLT_tau160_medium1_tracktwo",
    "HLT_tau35_medium1_tracktwo_tau25_medium1_tracktwo_L1TAU20IM_2TAU12IM"
]

# Global lepton trigger configuration.
# Can only specify triggers defined in https://gitlab.cern.ch/atlas/athena/-/blob/main/Trigger/TrigAnalysis/TrigGlobalEfficiencyCorrection/data/Triggers.cfg
triggerChainsPerYear = {
     '2015': ['HLT_e24_lhmedium_L1EM20VH || HLT_e60_lhmedium || HLT_e120_lhloose', 'HLT_mu20_iloose_L1MU15 || HLT_mu40'], 
     '2016': ['HLT_e26_lhtight_nod0_ivarloose || HLT_e60_lhmedium_nod0 || HLT_e140_lhloose_nod0', 'HLT_mu26_ivarmedium || HLT_mu50'] 
}

# global tau triggers
triggerChainsPerYearTau = {
    '2015': ['HLT_tau80L1TAU60_medium1_tracktwo','HLT_tau35_medium1_tracktwo','HLT_tau25_medium1_tracktwo'],
    '2016': ['HLT_tau160_medium1_tracktwo','HLT_tau35_medium1_tracktwo','HLT_tau25_medium1_tracktwo']
}

# List of good runs in 2015 and 2016 (i.e. runs passing certain detector and data quality requirements)
GRLs = ['GoodRunsLists/data15_13TeV/20170619/data15_13TeV.periodAllYear_DetStatus-v89-pro21-02_Unknown_PHYS_StandardGRL_All_Good_25ns.xml',
        'GoodRunsLists/data16_13TeV/20180129/data16_13TeV.periodAllYear_DetStatus-v89-pro21-01_DQDefects-00-02-04_PHYS_StandardGRL_All_Good_25ns.xml']

def makeSequence(dataType,metadata,flags):
    
    algSeq = AlgSequence()

    configSeq = ConfigSequence()

    # create factory object to build block configurations
    from AnalysisAlgorithmsConfig.ConfigFactory import ConfigFactory
    config = ConfigFactory()
    # get function to make configs
    makeConfig = config.makeConfig

    configSeq += makeConfig('CommonServices')

    # Configures the event cleaning and selection
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/AsgAnalysisAlgorithms/python/EventCleaningConfig.py
    configSeq += makeConfig('EventCleaning')
    configSeq.setOptionValue('.runEventCleaning', True)
    # Give it the list of Good Runs Lists (define above)
    configSeq.setOptionValue('.userGRLFiles', GRLs) 
    # If True: do apply event decoration, but do not filter
    # If False: only events passing the event cleaning/GRL etc. are selected
    configSeq.setOptionValue('.noFilter', False)

    if dataType is not DataType.Data:
        # Include, and then set up the generator analysis sequence:
        configSeq += makeConfig( 'GeneratorLevelAnalysis' )

    # Include and setup small R jet algorithm:
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/JetAnalysisAlgorithms/python/JetAnalysisConfig.py
    configSeq += makeConfig( 'Jets',
                             containerName='AnalysisJets',
                             jetCollection='AntiKt4EMPFlowJets')
    configSeq.setOptionValue('.runJvtUpdate', False)
    configSeq.setOptionValue('.runNNJvtUpdate', True)

    # Define the b-tagging working point (https://ftag.docs.cern.ch/recommendations/algs/r22-preliminary/#flavor-tagging-recommendations-r22)
    btagger = "DL1dv01"
    btagWP = "Continuous"
    # Include and setup b-jet algorithm
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/FTagAnalysisAlgorithms/python/FTagAnalysisConfig.py
    configSeq += makeConfig( 'Jets.FlavourTagging',
                             containerName='AnalysisJets')
    configSeq.setOptionValue('.btagger', btagger)
    configSeq.setOptionValue('.btagWP', btagWP)
    configSeq.setOptionValue('.bTagCalibFile','xAODBTaggingEfficiency/13TeV/2023-22-13TeV-MC20-CDI-2023-09-13_v1.root')
    
    ## This is needed 
    configSeq += config.makeConfig( 'Jets.FlavourTaggingEventSF',
                                    containerName='AnalysisJets')
    configSeq.setOptionValue('.btagger', btagger)
    configSeq.setOptionValue('.btagWP', btagWP)
    configSeq.setOptionValue('.bTagCalibFile','xAODBTaggingEfficiency/13TeV/2023-22-13TeV-MC20-CDI-2023-09-13_v1.root')
    # The only Sherpa generator versions which are supported
    # in the recomended CDI file for DL1Dv0 tagger are
    # 2.2.1, 2.2.10 and 2.2.12 so need to adjust. If any version
    # of Sherpa <= 8 configuring with Sherpa221.
    # The versions >= 10 are handled correctly internally
    # by the tool so can use autoconfig for the generator (default)
    if metadata.get("generators", "") in ['Sherpa(v.2.2.2)','Sherpa(v.2.2.3)','Sherpa(v.2.2.4)','Sherpa']: # Sherpa228 has only Sherpa
        configSeq.setOptionValue('.generator','Sherpa221')
    configSeq.setOptionValue('.savePerJetSF',True)
    # Include and setup large R jet algorithm
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/JetAnalysisAlgorithms/python/JetAnalysisConfig.py
    configSeq += makeConfig( 'Jets',
                             containerName='AnalysisLargeRJets',
                             jetCollection='AntiKt10UFOCSSKSoftDropBeta100Zcut10Jets')

    # Include and setup electron algorithm
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/EgammaAnalysisAlgorithms/python/ElectronAnalysisConfig.py
    configSeq += makeConfig( 'Electrons',
                             containerName='AnalysisElectrons')
    
    # We want to know if the electrons passed various configurations
    # of identification and isolation requirements (see: https://atlas-iff.docs.cern.ch/rel22recommendedisowps/index.html).
    # We need to create one config for each of these and give them unique names so that the
    # decorated variables can be retrieved in the loop over electrons later (in LeptonInfo.cxx)
    
    # ID: LooseBLayerLH, isolation: Loose_VarRad
    configSeq += makeConfig( 'Electrons.WorkingPoint',
                             containerName='AnalysisElectrons',
                             selectionName='electron_loose')
    configSeq.setOptionValue('.identificationWP','LooseBLayerLH')
    configSeq.setOptionValue('.isolationWP','Loose_VarRad')
    configSeq.setOptionValue('.recomputeID', False)
    # Needed as long as electron SFs are not avaialble (MR https://gitlab.cern.ch/atlas/athena/-/merge_requests/72590)
    configSeq.setOptionValue('.noEffSF',True) 

    # ID: LooseBLayerLH, isolation: Tight_VarRad
    configSeq += makeConfig( 'Electrons.WorkingPoint',
                             containerName='AnalysisElectrons',
                             selectionName='electron_tightiso')
    configSeq.setOptionValue('.identificationWP','LooseBLayerLH')
    configSeq.setOptionValue('.isolationWP','Tight_VarRad')
    configSeq.setOptionValue('.recomputeID', False)
    configSeq.setOptionValue('.noEffSF',True)

    # Include and setup algorithm for getting electron trigger scale factors
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/EgammaAnalysisAlgorithms/python/ElectronAnalysisConfig.py
    configSeq += makeConfig( 'Electrons.TriggerSF')
    configSeq.setOptionValue('.containerName','AnalysisElectrons')
    configSeq.setOptionValue('.electronID','TightLH')
    # Tight_VarRad not available for trigger scalefactor, but almost identical is FCTight
    configSeq.setOptionValue('.electronIsol','FCTight') 
    configSeq.setOptionValue('.triggerChainsPerYear', triggerChainsPerYear)

    # Include and setup photon algorithm
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/EgammaAnalysisAlgorithms/python/PhotonAnalysisConfig.py
    configSeq += makeConfig( 'Photons',
                             containerName='AnalysisPhotons' )
    configSeq.setOptionValue('.forceFullSimConfig', False)
    configSeq.setOptionValue('.recomputeIsEM', False)

    # We want to know if the photons passed various configurations
    # of identification and isolation requirements (see: https://atlas-iff.docs.cern.ch/rel22recommendedisowps/index.html).
    # We need to create one config for each of these and give them unique names (using selectionName option) so that the
    # decorated variables can be retrieved in the loop over photons later (in PhotonInfo.cxx)
    
    configSeq += makeConfig( 'Photons.WorkingPoint',
                             containerName='AnalysisPhotons',
                             selectionName='photon_loose')
    # ID: Loose, isolation: FixedCutLoose
    configSeq.setOptionValue('.qualityWP', 'Loose')
    configSeq.setOptionValue('.isolationWP', 'FixedCutLoose')
    configSeq.setOptionValue('.forceFullSimConfig', False)
    configSeq.setOptionValue('.recomputeIsEM', False)

    # ID: Tight, isolation: FixedCutLoose
    configSeq += makeConfig( 'Photons.WorkingPoint',
                             containerName='AnalysisPhotons',
                             selectionName='photon_tight')
    configSeq.setOptionValue('.qualityWP', 'Tight')
    configSeq.setOptionValue('.isolationWP', 'FixedCutLoose')
    configSeq.setOptionValue('.forceFullSimConfig', False)
    configSeq.setOptionValue('.recomputeIsEM', False)

    # ID: Loose, isolation: FixedCutTight
    configSeq += makeConfig( 'Photons.WorkingPoint',
                             containerName='AnalysisPhotons',
                             selectionName='photon_tightiso')
    configSeq.setOptionValue('.qualityWP', 'Loose')
    configSeq.setOptionValue('.isolationWP', 'FixedCutTight')
    configSeq.setOptionValue('.forceFullSimConfig', False)
    configSeq.setOptionValue('.recomputeIsEM', False)

    # Include and setup muons algorithm
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/MuonAnalysisAlgorithms/python/MuonAnalysisConfig.py
    configSeq += makeConfig( 'Muons',
                             containerName='AnalysisMuons' )

    # We want to know if the muons passed various configurations
    # of identification and isolation requirements (see: https://atlas-iff.docs.cern.ch/rel22recommendedisowps/index.html).
    # We need to create one config for each of these and give them unique names (using selectionName option) so that the
    # decorated variables can be retrieved in the loop over muons later (in LeptonInfo.cxx)

    # ID: Medium, isolation: Loose_VarRad
    configSeq += makeConfig( 'Muons.WorkingPoint',
                             containerName='AnalysisMuons',
                             selectionName='medium' )
    configSeq.setOptionValue('.quality', 'Medium')
    configSeq.setOptionValue('.isolation', 'Loose_VarRad')

    # ID: Medium, isolation: Tight_VarRad
    configSeq += makeConfig( 'Muons.WorkingPoint',
                             containerName='AnalysisMuons',
                             selectionName='tightiso' )
    configSeq.setOptionValue('.quality', 'Medium')
    configSeq.setOptionValue('.isolation', 'Tight_VarRad')

    # ID: Tight, isolation: Loose_VarRad
    configSeq += makeConfig( 'Muons.WorkingPoint',
                             containerName='AnalysisMuons',
                             selectionName='tight' )
    configSeq.setOptionValue('.quality', 'Tight')
    configSeq.setOptionValue('.isolation', 'Loose_VarRad')

    # ID: Loose, isolation: Loose_VarRad
    configSeq += makeConfig( 'Muons.WorkingPoint',
                             containerName='AnalysisMuons',
                             selectionName='loose' )
    configSeq.setOptionValue('.quality', 'Loose')
    configSeq.setOptionValue('.isolation', 'Loose_VarRad')

    # Include and setup algorithm for getting muon trigger scale factors
    configSeq += makeConfig( 'Muons.TriggerSF')
    configSeq.setOptionValue('.containerName','AnalysisMuons')
    configSeq.setOptionValue('.muonID', 'Medium')
    configSeq.setOptionValue('.triggerChainsPerYear', triggerChainsPerYear)

    # Include and setup tau algorithm
    configSeq += makeConfig( 'TauJets',
                             containerName='AnalysisTauJets' )

    # The working point to use for taus is Tight
    configSeq += makeConfig( 'TauJets.WorkingPoint',
                             containerName='AnalysisTauJets',
                             selectionName='tight' )
    configSeq.setOptionValue('.quality', 'Tight')

    # Include and setup algorithm for getting tau trigger scale factors
    configSeq += makeConfig( 'TauJets.TriggerSF')
    configSeq.setOptionValue('.containerName', 'AnalysisTauJets')
    configSeq.setOptionValue('.tauID', 'Tight')
    configSeq.setOptionValue('.triggerChainsPerYear',triggerChainsPerYearTau)

    # The following configurations define decorations to every object (electrons, photons, muons, etc.)
    # to reflect if the pT and eta thresholds defined at the top are satisfied. These are used in the code
    # to add a basic requirement on the object to select in the ntuples. Adds a decorator "selectPtEta" for
    # each object
    configSeq += makeConfig( 'Electrons.PtEtaSelection',
                             containerName='AnalysisElectrons' )
    configSeq.setOptionValue('.selectionDecoration', 'selectPtEta')
    configSeq.setOptionValue('.minPt', electronMinPt)
    configSeq.setOptionValue('.maxEta', electronMaxEta)

    configSeq += makeConfig( 'Photons.PtEtaSelection',
                             containerName='AnalysisPhotons' )
    configSeq.setOptionValue('.selectionDecoration', 'selectPtEta')
    configSeq.setOptionValue('.minPt', photonMinPt)
    configSeq.setOptionValue('.maxEta', photonMaxEta)

    configSeq += makeConfig( 'Muons.PtEtaSelection',
                             containerName='AnalysisMuons' )
    configSeq.setOptionValue('.selectionDecoration', 'selectPtEta')
    configSeq.setOptionValue('.minPt', muonMinPt)
    configSeq.setOptionValue('.maxEta', muonMaxEta)

    configSeq += makeConfig( 'Jets.PtEtaSelection',
                             containerName='AnalysisJets' )
    configSeq.setOptionValue('.selectionDecoration', 'selectPtEta')
    configSeq.setOptionValue('.minPt', jetMinPt)
    configSeq.setOptionValue('.maxEta', jetMaxEta)

    configSeq += makeConfig( 'Jets.PtEtaSelection',
                             containerName='AnalysisLargeRJets' )
    configSeq.setOptionValue('.selectionDecoration', 'selectPtEtaMine')
    configSeq.setOptionValue('.minPt', largeRJetMinPt)
    configSeq.setOptionValue('.maxEta', largeRJetMaxEta)

    configSeq += makeConfig( 'TauJets.PtEtaSelection',
                             containerName='AnalysisTauJets' )
    configSeq.setOptionValue('.selectionDecoration', 'selectPtEta')
    configSeq.setOptionValue('.minPt', tauMinPt)
    configSeq.setOptionValue('.maxEta', tauMaxEta)


    # Include and setup met algorithm:
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/MetAnalysisAlgorithms/python/MetAnalysisConfig.py
    configSeq += makeConfig( 'MissingET',
                             containerName='AnalysisMET' )
    configSeq.setOptionValue('.jets', 'AnalysisJets')
    configSeq.setOptionValue('.taus', 'AnalysisTauJets')
    configSeq.setOptionValue('.electrons', 'AnalysisElectrons')
    configSeq.setOptionValue('.photons', 'AnalysisPhotons')
    configSeq.setOptionValue('.muons', 'AnalysisMuons')

    # Include and setup overlap removal (OR) algorithm:
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/AsgAnalysisAlgorithms/python/OverlapAnalysisConfig.py
    # The current implementation follows the default, but there are many possibilities to adjust these by setting various flags
    # (see config file above). The current implementation apply the following OR:
    
    # doEleMuOR     : reject electrons if share track with muon
    # doEleJetOR    : reject electron if within deltaR < 0.4 of jet
    # doMuJetOR     : reject muon if within deltaR < 0.4 of jet
    # doTauEleOR    : reject tau if within deltaR < 0.2 of electron
    # doTauMuOR     : reject tau if within deltaR < 0.2 of muon
    # doTauJetOR    : reject jet if within deltaR < 0.2 of tau
    # doPhEleOR     : reject photon if within deltaR < 0.4 of electron
    # doPhMuOR      : reject photon if within deltaR < 0.4 of muon
    # doPhJetOR     : reject photon if within deltaR < 0.4 of jet
    # doEleFatJetOR : reject large-R jet if within deltaR < 1.0 of an electron
    # doJetFatJetOR : reject small-R jet if within deltaR < 1.0 of a large-R jet
    
    # Adds two decorators "preselectOR" and "passesOR" for each object to reflect
    # whether or not the object passed the OR
    configSeq += makeConfig( 'OverlapRemoval' )
    configSeq.setOptionValue('.electrons', 'AnalysisElectrons')
    configSeq.setOptionValue('.photons', 'AnalysisPhotons')
    configSeq.setOptionValue('.muons', 'AnalysisMuons')
    configSeq.setOptionValue('.jets', 'AnalysisJets')
    configSeq.setOptionValue('.taus', 'AnalysisTauJets')
    configSeq.setOptionValue('.fatJets', 'AnalysisLargeRJets')
    configSeq.setOptionValue('.inputLabel', 'preselectOR')
    configSeq.setOptionValue('.outputLabel', 'passesOR')
    
    # Include and setup trigger algorithm:
    # https://gitlab.cern.ch/atlas/athena/-/blob/main/PhysicsAnalysis/Algorithms/TriggerAnalysisAlgorithms/python/TriggerAnalysisConfig.py
    configSeq += makeConfig( 'Trigger')
    configSeq.setOptionValue('.triggerChainsForSelection', triggerChains)
    configSeq.setOptionValue('.triggerMatchingChainsPerYear',triggerChainsPerYear)
    configSeq.setOptionValue('.electronID', 'Tight')
    configSeq.setOptionValue('.electronIsol', 'FCTight')
    configSeq.setOptionValue('.photonIsol', 'FixedCutTight')
    configSeq.setOptionValue('.muonID', 'Medium')
    configSeq.setOptionValue('.electrons', 'AnalysisElectrons')
    configSeq.setOptionValue('.muons', 'AnalysisMuons')
    configSeq.setOptionValue('.photons', 'AnalysisPhotons')
    configSeq.setOptionValue('.taus', 'AnalysisTaus')
    configSeq.setOptionValue('.triggerChainsPerYear',triggerChainsPerYear)
    # Add the decorations, but no filtering done based on the triggers
    configSeq.setOptionValue('.noFilter',True)
    
    configSeq.printOptions()
    
    configAccumulator = ConfigAccumulator (algSeq, dataType, isPhyslite=True, campaign=Campaign.MC20a, geometry=LHCPeriod.Run2, noSystematics=True, autoconfigFromFlags=flags)
    configSeq.fullConfigure(configAccumulator)

    return algSeq

    # Do not add anything here!!!
