#!/usr/bin/env python3
from AthenaConfiguration.AllConfigFlags import initConfigFlags

# Read the submission directory as a command line argument. You can
# extend the list of arguments with your private ones later on.
import optparse
parser = optparse.OptionParser()
parser.add_option( '-s', '--submission-dir', dest = 'submission_dir',
                   action = 'store', type = 'string', default = 'submitDir',
                   help = 'Submission directory for EventLoop' )
parser.add_option( '-i', '--input-list', dest = 'input_list',
                   action = 'store', type = 'string', default = '',
                   help = 'Input file list; comma-separated list of files on each line')
parser.add_option( '-n', '--nEvents', dest = 'nEvents',
                   action = 'store', type = 'int', default = -1,
                   help = 'Number of events to process (-1 for all events)')
( options, args ) = parser.parse_args()

# Require an input file list
if options.input_list == '':
    print("No input list specified!")
    exit()

# Grab ROOT and initialize some xAOD ireading classes
import ROOT
ROOT.xAOD.Init().ignore()

# Set up the sample handler object.
import os
sh = ROOT.SH.SampleHandler()
# The tree containing event data in AOD files is called "CollectionTree"
sh.setMetaString( 'nc_tree', 'CollectionTree' )

# Output file name. Trees will go in {sample_name}.root, histograms into hist-{sample_name}.root
sample_name = 'output'
sample = ROOT.SH.SampleLocal(sample_name)

# Create a list of input files to run over based on the user input file
# Note that the inputs should be a single type (e.g. all full simulation) for the configuration to
#  work properly.
files = []
print("Start read")
with open(options.input_list) as fin:
    for aline in fin.readlines():
        line = aline.split('#')[0]
        files += [x.strip() for x in line.split(',')]
print(f"Read {len(files)} files from {options.input_list}")

# Add all the files to be processed to the sample, then add the sample to the sample handler
for f in files:
    sample.add(f)
sh.add(sample)

# Get some metadata from the input files in order to set Data / Simulation and Fast / Full simulation
flags = initConfigFlags()
flags.Input.Files = files
flags.lock()

from AthenaConfiguration.AutoConfigFlags import GetFileMD
metadata = GetFileMD(files)
dataType = ''
eventTypes = metadata.get('eventTypes', None)
if eventTypes:
    if 'IS_DATA' in eventTypes:
        dataType = 'data'
    elif 'IS_SIMULATION' in eventTypes:
        simFlavour = metadata['SimulationFlavour']
        if simFlavour is None:
            dataType = 'data'
        elif 'FullG4' in simFlavour:
            dataType = 'mc'
        elif 'ATLFAST3' in simFlavour:
            dataType = 'afii'
if ''==dataType:
    print(f'Warning: Could not identify dataType from {eventTypes=}')

#generator = metadata.get("generators", "")
#print(generator)

# Create an EventLoop job to process the samples in the sample handler
job = ROOT.EL.Job()
job.sampleHandler( sh )
# Create a unique output directory called submitDir-{time stamp}
# and add a link called submitDir that points to the most recent output directory
job.options().setString( ROOT.EL.Job.optSubmitDirMode, 'unique-link')
if options.nEvents>0:
    job.options().setDouble(ROOT.EL.Job.optMaxEvents, options.nEvents)

# Set up the sequence of algorithms to run on the input PHYSLITE
# This is a pretty standard set of physics object algorithms for calibrations etc
from PhysLiteToOpenData.PhysLiteToOpenDataAlgorithms import makeSequence
algSeq = makeSequence (dataType,metadata,flags)
# Add the sequence to the job
algSeq.addSelfToJob( job )

# Create the PhysLiteToOpenData algorithm's configuration.
# This algorithm creates output with a useful set of branches and histograms
from AnaAlgorithm.DualUseConfig import createAlgorithm
alg = createAlgorithm ( 'PhysLiteToOpenData', 'AnalysisAlg' )
# Configure our algorithm to use the data type that we have in the input
alg.dataType = dataType

# Add our algorithm to the job
job.algsAdd( alg )

# Define the output stream, which will be the name of the tree in our output file
job.outputAdd (ROOT.EL.OutputStream ('ANALYSIS'))

# Run the job using the direct driver (this means 'run the job here and now, locally')
# as opposed to a batch driver or grid driver (for a batch system or the Grid)
driver = ROOT.EL.DirectDriver()
driver.submit( job, options.submission_dir )
