<!--
SPDX-FileCopyrightText: 2021
- Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
- Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
- Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
- Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)

SPDX-License-Identifier: EUPL-1.2
-->

# S3 Storage

### Basic concepts

S3 is a storage service provided by Amazon in its cloud product Amazon Web Services (AWS).
It is merely a "key/value" repository and can be accessed via APIs.

S3 is based on just a few concepts:
1. **Buckets**: Buckets are "flat containers" that can hold an infinite number of objects.
2. **Objects**: Objects are binary blobs. It can be a file of any kind, text or binary based.
3. **Keys**: The key is a UTF-8 string used to identify an object.

{- However, only characters that are valid in URLs should be used.This is because the object is -} 
   {- accessed through REST APIs, where the object key is part of the URL. -}
   
Since it is a flat container, an S3 bucket has no hierarchy: it stores all objects 
at a single root level in the bucket. Nevertheless, it is possible to simulate a 
hierarchical structure by using the slash "/" character in the key. In this case, the AWS S3 
console displays such keys as a folder-like layout.

### Using the S3 storage module in code (minio-py)

`project.api.flask_minio.FlaskMinio` module has been created to allow uploading files to a S3 server,
it relies on MinIO (Python Client SDK) which is a Python Library for Amazon S3 Compatible Cloud Storage.
> The MinIO Python Client SDK provides simple APIs to access any Amazon 
> S3 compatible object storage server.

#### Initialize MinIO Client

You need these items in order to connect to MinIO object storage server.

| Params     | Description                                                      | 
| ---------- | ---------------------------------------------------------------- | 
| endpoint   | URL to object storage service.                                   | 
| access_key | Access key is like user ID that uniquely identifies your account.| 
| secret_key | Secret key is the password to your account.                      | 

#### Bucket policy

Bucket policy is a set of permissions that you can specify in a policy. each of which maps 
to a specific S3 [operation](https://docs.aws.amazon.com/AmazonS3/latest/API/API_Operations.html).

The following example bucket policy grants the `s3:GetObject` permissions to all user to be able 
to get a download url for an object inside (SMS) bucket as you can see in `Resource` [ARN](https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-arn-format.html).
With the `Principal` element, you can attach the policy to a user.

```json
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "*"
        ]
      },
      "Action": [
        "s3:GetBucketLocation",
        "s3:ListBucket"
      ],
      "Resource": [
        "arn:aws:s3:::sms"
      ]
    },
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "*"
        ]
      },
      "Action": [
        "s3:GetObject"
      ],
      "Resource": [
        "arn:aws:s3:::sms/*"
      ]
    }
  ]
}
```

### Example of how to use
upload takes only one parameter which is the file object.

```http 
POST /rdm/svm-api/v1/upload HTTP/1.1
Host: localhost:5000
Accept: application/vnd.api+json
Content-Type: multipart/form-data
Content-Disposition: form-data; name="file"; filename="<path_to_ur_file>"
```
Server response should be something like that.
```json
{
    "message": "object stored in sms-attachments",
    "url": "$HOST/<bucket_name>/<object_name>"
}
```
### DELETE Object

DELETE Object in the code is established by intercepting the delete operation and send a delete 
request to the minio server.
```python
def delete_attachments_in_minio_by_id(device_id_intended_for_deletion):
    """
    Use the minio class to delete an attachment or a list of attachments.
    :param id_intended_for_deletion:
    :return:
    """
    attachments_related_to_device = (
        db.session.query(DeviceAttachment)
        .filter_by(device_id=device_id_intended_for_deletion)
        .all()
    )
    for attachment in attachments_related_to_device:
        minio.remove_an_object(attachment.url)
```

```python 
def remove_an_object(self, object_path):
    """
    Removes an object form the Minio Storage by file path.
    :param object_path: a file url like:
        "http://172.16.238.10:9000/sms3/2021-02/860904cd822146b399a6d5ea48d39787.png"
    :return:
    """
    _bucket_name, _object_name = self.extract_bucket_and_file_names_from_url(
        object_path
    )
    try:
        self.connection.remove_object(_bucket_name, _object_name)
        return make_response(f"{_object_name} has been successfully removed")
    except ResponseError as err:
        raise JsonApiException({'parameter': err})

```
### Get File Metadata

Responsible uploader is stored in file metadata to get it please take a look at this example:

```python
from minio import Minio
from minio.error import S3Error


def main():
    # Create a client with the MinIO server playground, its access key
    # and secret key.
    client = Minio(
        "localhost:9000",
        access_key="admin",
        secret_key="minio123",
        secure=False
    )

    # result = client.stat_object("my-bucket", "my-object")
    result = client.stat_object("sms-attachments", "2021-08/57f1eed8e7884b0b861274a497be85c0.pdf")
    print(
        "last-modified: {0}, uploaded by id: {1}".format(
            result.last_modified
            , result.metadata['x-amz-meta-uploaded-by-id']
        ),
    )


if __name__ == "__main__":
    try:
        main()
    except S3Error as exc:
        print("error occurred.", exc)

```


## References
- [Getting started with Amazon S3](https://docs.aws.amazon.com/AmazonS3/latest/userguide/GetStartedWithS3.html)
- [Developer guide](https://docs.aws.amazon.com/AmazonS3/latest/userguide/Welcome.html)
- [Permissions](https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-with-s3-actions.html)
- [MinIO Quickstart](https://docs.min.io/docs/minio-quickstart-guide.html)
- [MinIO Admin](https://docs.min.io/docs/minio-admin-complete-guide.html)
- [minio-py](https://github.com/minio/minio-py)

#TOC