# SPDX-FileCopyrightText: 2021 - 2024
# - Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
# - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
# - Luca Johannes Nendel <luca-johannes.nendel@ufz.de>
# - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
# - Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)
#
# SPDX-License-Identifier: EUPL-1.2

"""Tests for the software types."""

from django.conf import settings
from django.test import TestCase
from django.urls import reverse
from django.utils import encoding

from app.models import GlobalProvenance, SoftwareType
from app.tests.Admin_class import TestAdminInterface
from app.tests.Base_class import BaseTestCase


class SoftwareTypeTestCase(TestCase):
    """Test case for the software types."""

    list_url = reverse("softwaretype-list")

    def setUp(self):
        """Set up some test data."""
        gl = GlobalProvenance.objects.create(
            id=1,
            name="test global provenance",
            description="test global provenance description",
            uri="test global provenance uri",
        )
        gl.save()

        st = SoftwareType.objects.create(
            id=1,
            term="Test SoftwareType",
            definition="Test definition",
            provenance=None,
            provenance_uri=None,
            category="Test",
            note="Test1",
            global_provenance_id=1,
            successor_id=None,
        )
        st.save()
        self.detail_url = reverse("softwaretype-detail", kwargs={"pk": st.pk})

    def test_term(self):
        """Test the term property."""
        st = SoftwareType.objects.get(id=1)
        self.assertEqual(st.term, "Test SoftwareType")

    def test_global_provenance(self):
        """Test the global provenance property."""
        st = SoftwareType.objects.get(id=1)
        self.assertEqual(st.global_provenance_id, 1)

    def test_get_all(self):
        """Ensure the result has all attributes in 'SoftwareType'."""
        response = self.client.get(self.list_url)
        self.assertEqual(response.status_code, 200)

        st = SoftwareType.objects.all()[0]
        expected = {
            "links": {
                "first": "".join(
                    [
                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                        "softwaretypes/?page%5Bnumber%5D=1",
                    ]
                ),
                "last": "".join(
                    [
                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                        "softwaretypes/?page%5Bnumber%5D=1",
                    ]
                ),
                "next": None,
                "prev": None,
            },
            "data": [
                {
                    "type": "SoftwareType",
                    "id": encoding.force_str(st.pk),
                    "attributes": {
                        "term": st.term,
                        "definition": st.definition,
                        "provenance": st.provenance,
                        "provenance_uri": st.provenance_uri,
                        "category": st.category,
                        "note": st.note,
                        "status": st.status,
                        "requested_by_email": st.requested_by_email,
                        "discussion_url": st.discussion_url,
                    },
                    "relationships": {
                        "global_provenance": {
                            "links": {
                                "self": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "softwaretypes/1/relationships/global_provenance",
                                    ]
                                ),
                                "related": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "softwaretypes/1/global_provenance/",
                                    ]
                                ),
                            },
                            "data": {
                                "type": "GlobalProvenance",
                                "id": encoding.force_str(st.global_provenance_id),
                            },
                        },
                        "successor": {
                            "links": {
                                "self": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "softwaretypes/1/relationships/successor",
                                    ]
                                ),
                                "related": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "softwaretypes/1/successor/",
                                    ]
                                ),
                            }
                        },
                    },
                    "links": {
                        "self": "".join(
                            [
                                f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                "softwaretypes/1/",
                            ]
                        ),
                    },
                }
            ],
            "meta": {"pagination": {"page": 1, "pages": 1, "count": 1}},
        }
        assert expected == response.json()

    def test_sort(self):
        """Test the ordering."""
        BaseTestCase().sort(self.list_url, self.client)

    def test_admin(self):
        """Test the admin interface."""
        obj = SoftwareType.objects.get(id=1)
        test_admin = TestAdminInterface().setUpAdmin(self.client, obj)
        # test response list view
        test_admin.list_view_responding()
        # test response change view
        test_admin.add_view_responding()
        # test response change view
        test_admin.change_view_responding()
        # test response delete view
        # test_admin.delete_view_responding()
