# SPDX-FileCopyrightText: 2022 - 2024
# - Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
# - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
# - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
# - Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)
#
# SPDX-License-Identifier: EUPL-1.2

"""Configuration resource classes."""

import os

from flask import g, request
from flask_rest_jsonapi import JsonApiException, ResourceDetail, ResourceList
from flask_rest_jsonapi.exceptions import ObjectNotFound
from sqlalchemy.orm.exc import NoResultFound

from ...extensions.instances import pidinst
from ..datalayers.esalchemy import (
    AndFilter,
    EsSqlalchemyDataLayer,
    TermEqualsExactStringFilter,
)
from ..helpers.db import save_to_db
from ..helpers.errors import ConflictError, ForbiddenError
from ..helpers.resource_mixin import add_created_by_id, add_updated_by_id
from ..models import ActivityLog, Configuration, ConfigurationContactRole, Site
from ..models.base_model import db
from ..permissions.common import DelegateToCanFunctions
from ..permissions.rules import filter_visible, filter_visible_es
from ..schemas.configuration_schema import ConfigurationSchema
from .base_resource import check_if_object_not_found, delete_attachments_in_minio_by_url
from .mixins.mqtt_notification import MqttNotificationMixin


class ConfigurationList(MqttNotificationMixin, ResourceList):
    """
    Resource for the configuration list endpoint.

    Supports GET (list) & POST (create) methods.
    """

    def query(self, view_kwargs):
        """
        Filter for what the user is allowed to query.

        :param view_kwargs:
        :return: queryset or es filter
        """
        query = filter_visible(self.session.query(self.model))
        site_id = view_kwargs.get("site_id")
        if site_id is not None:
            try:
                self.session.query(Site).filter_by(id=site_id).one()
            except NoResultFound:
                raise ObjectNotFound(
                    {"parameter": "id"}, "Site: {} not found".format(site_id)
                )
            else:
                query = query.filter(Configuration.site_id == site_id)

        false_values = ["false"]
        # hide archived must be disabled explicitly
        hide_archived = request.args.get("hide_archived") not in false_values
        if hide_archived:
            query = query.filter_by(archived=False)
        return query

    def es_query(self, view_kwargs):
        """
        Return the elasticsearch filter for the query.

        Should return the same set as query, but using
        the elasticsearch fields.
        """
        and_filters = [filter_visible_es(self.model)]

        site_id = view_kwargs.get("site_id")
        if site_id is not None:
            and_filters.append(TermEqualsExactStringFilter("site_id", site_id))

        false_values = ["false"]
        # hide archived must be disabled explicitly
        hide_archived = request.args.get("hide_archived") not in false_values
        if hide_archived:
            and_filters.append(TermEqualsExactStringFilter("archived", False))

        return AndFilter.combine_optionals(and_filters)

    def before_create_object(self, data, *args, **kwargs):
        """
        Set the visibility of the object (internal of nothing else is given).

        :param data: data of the request (as dict)
        :param args:
        :param kwargs:
        :return: None
        """
        # Will modify the data inplace.
        if not any([data.get("is_public"), data.get("is_internal")]):
            data["is_internal"] = True
            data["is_public"] = False
        add_created_by_id(data)

    def after_post(self, result):
        """
        Automatically add the created user to object contacts.

        Also add the owner to contact role.

        :param result:
        :return:
        """
        result_id = result[0]["data"]["id"]
        configuration = db.session.query(Configuration).filter_by(id=result_id).first()
        contact = g.user.contact
        cv_url = os.environ.get("CV_URL")
        role_name = "Owner"
        role_uri = f"{cv_url}/contactroles/4/"
        contact_role = ConfigurationContactRole(
            contact_id=contact.id,
            configuration_id=configuration.id,
            role_name=role_name,
            role_uri=role_uri,
        )
        db.session.add(contact_role)

        msg = "create;basic data"
        configuration.update_description = msg
        configuration.updated_by_id = g.user.id

        save_to_db(configuration)

        new_log_entry = ActivityLog.create(
            entity=configuration,
            user=g.user,
            description=msg,
        )
        save_to_db(new_log_entry)

        return result

    schema = ConfigurationSchema
    data_layer = {
        "session": db.session,
        "model": Configuration,
        "class": EsSqlalchemyDataLayer,
        "methods": {
            "before_create_object": before_create_object,
            "query": query,
            "es_query": es_query,
        },
    }
    permission_classes = [DelegateToCanFunctions]


class ConfigurationDetail(MqttNotificationMixin, ResourceDetail):
    """
    Detail resource class for the configurations.

    Provides get, patch and delete methods to retrieve details
    of an object, update an object and delete a Configuration.
    """

    def before_get(self, args, kwargs):
        """Prevent not registered users form viewing internal configs."""
        check_if_object_not_found(self._data_layer.model, kwargs)

    def before_patch(self, args, kwargs, data):
        """Check if a user has the permission to change this configuration."""
        add_updated_by_id(data)

    def after_patch(self, result):
        """Run some updates after the successful patch."""
        result_id = result["data"]["id"]
        configuration = db.session.query(Configuration).filter_by(id=result_id).first()
        msg = "update;basic data"
        configuration.update_description = msg

        save_to_db(configuration)

        new_log_entry = ActivityLog.create(
            entity=configuration,
            user=g.user,
            description=msg,
        )
        save_to_db(new_log_entry)

        if pidinst.has_external_metadata(configuration):
            pidinst.update_external_metadata(configuration)
        return result

    def delete(self, *args, **kwargs):
        """
        Try to delete an object through sqlalchemy.

        If could not be done give a ConflictError.
        :param args: args from the resource view
        :param kwargs: kwargs from the resource view
        :return:
        """
        configuration = check_if_object_not_found(Configuration, kwargs)

        urls = [
            a.internal_url
            for a in configuration.configuration_attachments
            if a.internal_url
        ]
        try:
            super().delete(*args, **kwargs)
        except ForbiddenError as e:
            # Just re-raise it
            raise e
        except JsonApiException as e:
            raise ConflictError("Deletion failed for the configuration.", str(e))

        new_log_entry = ActivityLog.create(
            entity=configuration, user=g.user, description="delete;basic data", data={}
        )
        save_to_db(new_log_entry)

        for url in urls:
            delete_attachments_in_minio_by_url(url)

        final_result = {"meta": {"message": "Object successfully deleted"}}
        return final_result

    schema = ConfigurationSchema
    data_layer = {
        "session": db.session,
        "model": Configuration,
    }
    permission_classes = [DelegateToCanFunctions]
