# SPDX-FileCopyrightText: 2021 - 2024
# - Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
# - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
# - Luca Johannes Nendel <luca-johannes.nendel@ufz.de>
# - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
# - Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)
#
# SPDX-License-Identifier: EUPL-1.2

"""Tests for the dynamic location actions."""

import datetime
import json

import dateutil.parser
import pytz

from project import base_url, db
from project.api.models import (
    Configuration,
    ConfigurationDynamicLocationBeginAction,
    Contact,
    Device,
    DeviceMountAction,
    DeviceProperty,
    User,
)
from project.extensions.instances import mqtt
from project.tests.base import (
    BaseTestCase,
    Fixtures,
    create_token,
    fake,
    generate_userinfo_data,
)
from project.tests.models.test_configuration_dynamic_action_model import (
    add_dynamic_location_begin_action_model,
)
from project.tests.models.test_configurations_model import generate_configuration_model

fixtures = Fixtures()


@fixtures.register("public_configuration1_in_group1", scope=lambda: db.session)
def create_public_configuration1_in_group1():
    """Create a public configuration that uses group 1 for permission management."""
    result = Configuration(
        label="public configuration1",
        is_internal=False,
        is_public=True,
        cfg_permission_group="1",
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register("public_device1_in_group1", scope=lambda: db.session)
def create_public_device1_in_group1():
    """Create a public device that uses group 1 for permission management."""
    result = Device(
        short_name="public device1",
        is_internal=False,
        is_public=True,
        group_ids=["1"],
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register(
    "device_property1_of_public_device1_in_group1", scope=lambda: db.session
)
@fixtures.use(["public_device1_in_group1"])
def create_device_property1_of_public_device1_in_group1(public_device1_in_group1):
    """Create a device property for the public device1."""
    result = DeviceProperty(
        device=public_device1_in_group1,
        property_name="AirTemperature",
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register("mount_of_public_device1_in_group1", scope=lambda: db.session)
@fixtures.use(
    [
        "public_device1_in_group1",
        "public_configuration1_in_group1",
        "super_user_contact",
    ]
)
def create_mount_of_public_device1_in_group1(
    public_device1_in_group1, public_configuration1_in_group1, super_user_contact
):
    """Create a mount for the public device1 of public configuration1."""
    result = DeviceMountAction(
        device=public_device1_in_group1,
        configuration=public_configuration1_in_group1,
        begin_contact=super_user_contact,
        begin_date=datetime.datetime(2018, 1, 1, 0, 0, 0, tzinfo=pytz.utc),
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register("super_user_contact", scope=lambda: db.session)
def create_super_user_contact():
    """Create a contact that can be used to make a super user."""
    result = Contact(
        given_name="super", family_name="contact", email="super.contact@localhost"
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register(
    "dynamic_location1_of_public_configuration1_in_group1", scope=lambda: db.session
)
@fixtures.use(["public_configuration1_in_group1", "super_user_contact"])
def create_dynamic_location1_of_public_configuration1_in_group1(
    public_configuration1_in_group1, super_user_contact
):
    """Create a dynamic location for the public configuration."""
    result = ConfigurationDynamicLocationBeginAction(
        configuration=public_configuration1_in_group1,
        begin_date=datetime.datetime(2022, 1, 25, 0, 0, 0, tzinfo=pytz.UTC),
        begin_contact=super_user_contact,
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register("super_user", scope=lambda: db.session)
@fixtures.use(["super_user_contact"])
def create_super_user(super_user_contact):
    """Create super user to use it in the tests."""
    result = User(
        contact=super_user_contact, subject=super_user_contact.email, is_superuser=True
    )
    db.session.add(result)
    db.session.commit()
    return result


class TestConfigurationDynamicLocationBeginActionServices(BaseTestCase):
    """Tests for the ConfigurationDynamicLocationBeginAction endpoint."""

    url = base_url + "/dynamic-location-actions"
    contact_url = base_url + "/contacts"
    object_type = "configuration_dynamic_location_action"

    def test_get_configuration_dynamic_location_action(self):
        """Ensure the List /configuration_dynamic_location_action route behaves correctly."""
        response = self.client.get(self.url)
        self.assertEqual(response.status_code, 200)
        # no data yet
        self.assertEqual(response.json["data"], [])

    def test_get_device_calibration_action_collection(self):
        """Test retrieve a collection of configuration_dynamic_location_action objects."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model(
            is_public=True, is_private=False, is_internal=False
        )
        with self.client:
            response = self.client.get(self.url)
        data = json.loads(response.data.decode())
        self.assertEqual(response.status_code, 200)
        self.assertEqual(
            dynamic_location_begin_action.begin_description,
            data["data"][0]["attributes"]["begin_description"],
        )

    def test_add_configuration_dynamic_begin_location_action(self):
        """Ensure a new configuration dynamic location can be added to the database."""
        device = Device(
            short_name="Device 555",
            manufacturer_name=fake.pystr(),
            is_public=False,
            is_private=False,
            is_internal=True,
        )
        x_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        y_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test y_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        z_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test z_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model()
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all(
            [device, x_property, y_property, z_property, contact, config]
        )
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": "2021-08-22T10:00:50.542Z",
                    "end_date": "2021-10-22T10:00:50.542Z",
                    "begin_description": "beginning",
                    "end_description": "finishing",
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "end_contact": {"data": {"type": "contact", "id": contact.id}},
                    "x_property": {
                        "data": {"type": "device_property", "id": x_property.id}
                    },
                    "y_property": {
                        "data": {"type": "device_property", "id": y_property.id}
                    },
                    "z_property": {
                        "data": {"type": "device_property", "id": z_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        # Make sure that we have the device mount action for the xyz properties.
        device_mount_action = DeviceMountAction(
            device=device,
            configuration=config,
            begin_contact=contact,
            begin_date=dateutil.parser.parse(data["data"]["attributes"]["begin_date"]),
        )
        db.session.add(device_mount_action)
        db.session.commit()

        result = super().add_object(
            url=self.url,
            data_object=data,
            object_type=self.object_type,
        )
        configuration_id = result["data"]["relationships"]["configuration"]["data"][
            "id"
        ]
        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        self.assertEqual(
            configuration.update_description, "create;dynamic location action"
        )

    def test_add_for_archived_configuration(self):
        """Ensure we can't add a location for an archived configuration."""
        device = Device(
            short_name="Device 555",
            is_public=False,
            is_private=False,
            is_internal=True,
        )
        x_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        y_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test y_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        z_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test z_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model()
        config.archived = True
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all(
            [device, x_property, y_property, z_property, contact, config]
        )
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": "2021-08-22T10:00:50.542Z",
                    "end_date": "2021-10-22T10:00:50.542Z",
                    "begin_description": "beginning",
                    "end_description": "finishing",
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "end_contact": {"data": {"type": "contact", "id": contact.id}},
                    "x_property": {
                        "data": {"type": "device_property", "id": x_property.id}
                    },
                    "y_property": {
                        "data": {"type": "device_property", "id": y_property.id}
                    },
                    "z_property": {
                        "data": {"type": "device_property", "id": z_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        # Make sure that we have the device mount action for the xyz properties.
        device_mount_action = DeviceMountAction(
            device=device,
            configuration=config,
            begin_contact=contact,
            begin_date=dateutil.parser.parse(data["data"]["attributes"]["begin_date"]),
        )
        db.session.add(device_mount_action)
        db.session.commit()

        _ = super().try_add_object_with_status_code(
            url=self.url, data_object=data, expected_status_code=403
        )

    def test_add_for_archived_device(self):
        """Ensure we can't add a location for an archived configuration."""
        device = Device(
            short_name="Device 555",
            is_public=False,
            is_private=False,
            is_internal=True,
            archived=True,
        )
        x_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        y_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test y_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        z_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test z_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model()
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all(
            [device, x_property, y_property, z_property, contact, config]
        )
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": "2021-08-22T10:00:50.542Z",
                    "end_date": "2021-10-22T10:00:50.542Z",
                    "begin_description": "beginning",
                    "end_description": "finishing",
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "end_contact": {"data": {"type": "contact", "id": contact.id}},
                    "x_property": {
                        "data": {"type": "device_property", "id": x_property.id}
                    },
                    "y_property": {
                        "data": {"type": "device_property", "id": y_property.id}
                    },
                    "z_property": {
                        "data": {"type": "device_property", "id": z_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        # Make sure that we have the device mount action for the xyz properties.
        device_mount_action = DeviceMountAction(
            device=device,
            configuration=config,
            begin_contact=contact,
            begin_date=dateutil.parser.parse(data["data"]["attributes"]["begin_date"]),
        )
        db.session.add(device_mount_action)
        db.session.commit()

        _ = super().try_add_object_with_status_code(
            url=self.url, data_object=data, expected_status_code=409
        )

    def test_add_configuration_dynamic_begin_location_action_without_mount_action(self):
        """
        Ensure POST fails if there is no mount action for the xyz properties.

        This is part of the more advanced validation & we want to make sure
        that those run.
        """
        device = Device(
            short_name="Device 555",
            is_public=False,
            is_private=False,
            is_internal=True,
        )
        x_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        y_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test y_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        z_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test z_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model()
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all(
            [device, x_property, y_property, z_property, contact, config]
        )
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": "2021-08-22T10:00:50.542Z",
                    "end_date": "2021-10-22T10:00:50.542Z",
                    "begin_description": "beginning",
                    "end_description": "finishing",
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "end_contact": {"data": {"type": "contact", "id": contact.id}},
                    "x_property": {
                        "data": {"type": "device_property", "id": x_property.id}
                    },
                    "y_property": {
                        "data": {"type": "device_property", "id": y_property.id}
                    },
                    "z_property": {
                        "data": {"type": "device_property", "id": z_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        access_headers = create_token()
        with self.client:
            response = self.client.post(
                self.url,
                data=json.dumps(data),
                content_type="application/vnd.api+json",
                headers=access_headers,
            )
        self.assertEqual(response.status_code, 409)

    def prepare_request_data_with_config(self, description):
        """Create some request data to add a location action."""
        device = Device(
            short_name="Device 555",
            manufacturer_name=fake.pystr(),
            is_public=False,
            is_private=False,
            is_internal=True,
        )

        config = generate_configuration_model(
            is_public=True, is_private=False, is_internal=False
        )
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all([device, contact, config])
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": "2021-08-22T10:00:50.542Z",
                    "end_date": "2021-10-23T10:00:50.542Z",
                    "begin_description": description,
                    "end_description": description,
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "end_contact": {"data": {"type": "contact", "id": contact.id}},
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        return data, config

    def test_update_configuration_dynamic_begin_location_action(self):
        """Ensure a configuration_dynamic_begin_location_action can be updated."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        # This block for the device mount actions is just to make sure
        # that all the device properties have mounts for the timepoints.
        device_mounts_by_device_ids = {}
        contact_instance = (
            db.session.query(Contact).filter_by(id=contact["data"]["id"]).one()
        )
        for device_property in [
            dynamic_location_begin_action.x_property,
            dynamic_location_begin_action.y_property,
            dynamic_location_begin_action.z_property,
        ]:
            if device_property:
                device = device_property.device
                device_id = device.id
                if device_id not in device_mounts_by_device_ids.keys():
                    mount_action = DeviceMountAction(
                        device=device,
                        configuration=dynamic_location_begin_action.configuration,
                        begin_date=dynamic_location_begin_action.begin_date,
                        begin_contact=contact_instance,
                    )
                    db.session.add(mount_action)
                    db.session.commit()
                    device_mounts_by_device_ids[device_id] = mount_action
        new_data = {
            "data": {
                "type": self.object_type,
                "id": dynamic_location_begin_action.id,
                "attributes": {
                    "end_description": "stopped",
                    "end_date": "2021-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": {"type": "contact", "id": contact["data"]["id"]}
                    },
                },
            }
        }

        result = super().update_object(
            url=f"{self.url}/{dynamic_location_begin_action.id}",
            data_object=new_data,
            object_type=self.object_type,
        )
        configuration_id = result["data"]["relationships"]["configuration"]["data"][
            "id"
        ]
        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        self.assertEqual(
            configuration.update_description, "update;dynamic location action"
        )

    def test_update_archived_configuration(self):
        """Ensure that we can't change for archived configurations."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        dynamic_location_begin_action.configuration.archived = True
        db.session.add(dynamic_location_begin_action.configuration)
        db.session.commit()

        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        # This block for the device mount actions is just to make sure
        # that all the device properties have mounts for the timepoints.
        device_mounts_by_device_ids = {}
        contact_instance = (
            db.session.query(Contact).filter_by(id=contact["data"]["id"]).one()
        )
        for device_property in [
            dynamic_location_begin_action.x_property,
            dynamic_location_begin_action.y_property,
            dynamic_location_begin_action.z_property,
        ]:
            if device_property:
                device = device_property.device
                device_id = device.id
                if device_id not in device_mounts_by_device_ids.keys():
                    mount_action = DeviceMountAction(
                        device=device,
                        configuration=dynamic_location_begin_action.configuration,
                        begin_date=dynamic_location_begin_action.begin_date,
                        begin_contact=contact_instance,
                    )
                    db.session.add(mount_action)
                    db.session.commit()
                    device_mounts_by_device_ids[device_id] = mount_action
        new_data = {
            "data": {
                "type": self.object_type,
                "id": dynamic_location_begin_action.id,
                "attributes": {
                    "end_description": "stopped",
                    "end_date": "2021-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": {"type": "contact", "id": contact["data"]["id"]}
                    },
                },
            }
        }

        _ = super().try_update_object_with_status_code(
            url=f"{self.url}/{dynamic_location_begin_action.id}",
            data_object=new_data,
            expected_status_code=403,
        )

    def test_update_archived_device(self):
        """Ensure that we can't change for archived devices."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        dynamic_location_begin_action.x_property.device.archived = True
        db.session.add(dynamic_location_begin_action.x_property.device)
        db.session.commit()
        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        # This block for the device mount actions is just to make sure
        # that all the device properties have mounts for the timepoints.
        device_mounts_by_device_ids = {}
        contact_instance = (
            db.session.query(Contact).filter_by(id=contact["data"]["id"]).one()
        )
        for device_property in [
            dynamic_location_begin_action.x_property,
            dynamic_location_begin_action.y_property,
            dynamic_location_begin_action.z_property,
        ]:
            if device_property:
                device = device_property.device
                device_id = device.id
                if device_id not in device_mounts_by_device_ids.keys():
                    mount_action = DeviceMountAction(
                        device=device,
                        configuration=dynamic_location_begin_action.configuration,
                        begin_date=dynamic_location_begin_action.begin_date,
                        begin_contact=contact_instance,
                    )
                    db.session.add(mount_action)
                    db.session.commit()
                    device_mounts_by_device_ids[device_id] = mount_action
        new_data = {
            "data": {
                "type": self.object_type,
                "id": dynamic_location_begin_action.id,
                "attributes": {
                    "end_description": "stopped",
                    "end_date": "2021-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": {"type": "contact", "id": contact["data"]["id"]}
                    },
                },
            }
        }

        _ = super().try_update_object_with_status_code(
            url=f"{self.url}/{dynamic_location_begin_action.id}",
            data_object=new_data,
            expected_status_code=409,
        )

    def test_update_configuration_dynamic_begin_location_action_fail(self):
        """Ensure we validate the mounts for the xzy properties."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        contact = Contact(
            family_name="Bob", given_name="Meister", email="bob.meister@localhost"
        )
        superuser = User(contact=contact, is_superuser=True, subject=contact.email)
        db.session.add_all([contact, superuser])
        db.session.commit()
        # We don't add the mount action here, so the patch request will fail.
        new_data = {
            "data": {
                "type": self.object_type,
                "id": dynamic_location_begin_action.id,
                "attributes": {
                    "end_description": "stopped",
                    "end_date": "2021-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": {
                            "type": "contact",
                            "id": contact.id,
                        }
                    },
                },
            }
        }

        with self.run_requests_as(superuser):
            with self.client:
                response = self.client.patch(
                    f"{self.url}/{dynamic_location_begin_action.id}",
                    data=json.dumps(new_data),
                    content_type="application/vnd.api+json",
                )
        self.assertEqual(response.status_code, 409)

    def test_update_configuration_dynamic_begin_location_action_set_end_contact_to_none(
        self,
    ):
        """Ensure that we can reset the end_contact if necessary."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact_result = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        contact = (
            db.session.query(Contact).filter_by(id=contact_result["data"]["id"]).one()
        )
        dynamic_location_begin_action.end_contact = contact
        db.session.add_all([dynamic_location_begin_action, contact])
        db.session.commit()
        # This block for the device mount actions is just to make sure
        # that all the device properties have mounts for the timepoints.
        device_mounts_by_device_ids = {}
        for device_property in [
            dynamic_location_begin_action.x_property,
            dynamic_location_begin_action.y_property,
            dynamic_location_begin_action.z_property,
        ]:
            if device_property:
                device = device_property.device
                device_id = device.id
                if device_id not in device_mounts_by_device_ids.keys():
                    mount_action = DeviceMountAction(
                        device=device,
                        configuration=dynamic_location_begin_action.configuration,
                        begin_date=dynamic_location_begin_action.begin_date,
                        begin_contact=contact,
                    )
                    db.session.add(mount_action)
                    db.session.commit()
                    device_mounts_by_device_ids[device_id] = mount_action
        new_data = {
            "data": {
                "type": self.object_type,
                "id": dynamic_location_begin_action.id,
                "attributes": {
                    "end_description": "stopped",
                    "end_date": "2021-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": None,
                    },
                },
            }
        }

        _ = super().update_object(
            url=f"{self.url}/{dynamic_location_begin_action.id}",
            data_object=new_data,
            object_type=self.object_type,
        )

    def test_delete_configuration_dynamic_begin_location_action(self):
        """Ensure a configuration_dynamic_begin_location_action can be deleted."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        configuration_id = dynamic_location_begin_action.configuration_id

        _ = super().delete_object(
            url=f"{self.url}/{dynamic_location_begin_action.id}",
        )
        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        self.assertEqual(
            configuration.update_description, "delete;dynamic location action"
        )

    def test_delete_archived_configuration(self):
        """Ensure we can't delete for an archived configuration."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        dynamic_location_begin_action.configuration.archived = True
        db.session.add(dynamic_location_begin_action.configuration)
        db.session.commit()

        access_headers = create_token()
        with self.client:
            response = self.client.delete(
                f"{self.url}/{dynamic_location_begin_action.id}",
                content_type="application/vnd.api+json",
                headers=access_headers,
            )
        self.assertEqual(response.status_code, 403)

    def test_delete_archived_device(self):
        """Ensure we can't delete for an archived device."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        dynamic_location_begin_action.x_property.device.archived = True
        db.session.add(dynamic_location_begin_action.x_property.device)
        db.session.commit()

        access_headers = create_token()
        with self.client:
            response = self.client.delete(
                f"{self.url}/{dynamic_location_begin_action.id}",
                content_type="application/vnd.api+json",
                headers=access_headers,
            )
        self.assertEqual(response.status_code, 409)

    def test_filtered_by_configuration(self):
        """Ensure that filter by a specific configuration works."""
        data1, config1 = self.prepare_request_data_with_config(
            "test dynamic_location_begin_action1"
        )

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, _ = self.prepare_request_data_with_config(
            "test dynamic_location_begin_action2"
        )

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        with self.client:
            response = self.client.get(
                self.url, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 2)
        # Test only for the first one
        with self.client:
            url_get_for_config1 = (
                base_url + f"/configurations/{config1.id}/dynamic-location-actions"
            )
            response = self.client.get(
                url_get_for_config1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test dynamic_location_begin_action1",
        )

    def test_filtered_by_configuration_id(self):
        """Ensure that filter by filter[configuration_id]."""
        data1, config1 = self.prepare_request_data_with_config(
            "test dynamic_location_begin_action1"
        )

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, _ = self.prepare_request_data_with_config(
            "test dynamic_location_begin_action2"
        )

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        # Test only for the first one
        with self.client:
            url_get_for_config1 = (
                base_url
                + f"/dynamic-location-actions?filter[configuration_id]={config1.id}"
            )
            response = self.client.get(
                url_get_for_config1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test dynamic_location_begin_action1",
        )

    def prepare_request_data_with_x_property(self, description):
        """Prepare some payloads to add/update an x property."""
        device = Device(
            short_name="Device 575",
            manufacturer_name=fake.pystr(),
            is_public=False,
            is_private=False,
            is_internal=True,
        )
        x_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model(
            is_public=True, is_private=False, is_internal=False
        )
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all([device, contact, config, x_property])
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": fake.future_datetime().__str__(),
                    "begin_description": description,
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "x_property": {
                        "data": {"type": "device_property", "id": x_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        return data, x_property

    def prepare_request_data_with_y_property(self, description):
        """Prepare some payloads to add/update an y property."""
        device = Device(
            short_name="Device 575",
            manufacturer_name=fake.pystr(),
            is_public=False,
            is_private=False,
            is_internal=True,
        )
        y_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model(
            is_public=True, is_private=False, is_internal=False
        )
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all([device, contact, config, y_property])
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": fake.future_datetime().__str__(),
                    "begin_description": description,
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "y_property": {
                        "data": {"type": "device_property", "id": y_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        return data, y_property

    def prepare_request_data_with_z_property(self, description):
        """Prepare some payloads to add/update an z property."""
        device = Device(
            short_name="Device 575",
            manufacturer_name=fake.pystr(),
            is_public=False,
            is_private=False,
            is_internal=True,
        )
        z_property = DeviceProperty(
            device=device,
            measuring_range_min=fake.pyfloat(),
            measuring_range_max=fake.pyfloat(),
            failure_value=fake.pyfloat(),
            accuracy=fake.pyfloat(),
            label=fake.pystr(),
            unit_uri=fake.uri(),
            unit_name=fake.pystr(),
            compartment_uri=fake.uri(),
            compartment_name=fake.pystr(),
            property_uri=fake.uri(),
            property_name="Test x_property",
            sampling_media_uri=fake.uri(),
            sampling_media_name=fake.pystr(),
        )
        config = generate_configuration_model(
            is_public=True, is_private=False, is_internal=False
        )
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all([device, contact, config, z_property])
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": fake.future_datetime().__str__(),
                    "begin_description": description,
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "z_property": {
                        "data": {"type": "device_property", "id": z_property.id}
                    },
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        return data, z_property

    def ensure_device_mount_action_exists(self, data1, x_property1):
        """Help to add a device mount action for the dynamic location property."""
        # As we use a device property here, we need to make sure that
        # its device is also mounted on the configuration.
        configuration = (
            db.session.query(Configuration)
            .filter_by(id=data1["data"]["relationships"]["configuration"]["data"]["id"])
            .one()
        )
        contact = (
            db.session.query(Contact)
            .filter_by(id=data1["data"]["relationships"]["begin_contact"]["data"]["id"])
            .one()
        )
        device_mount_action = DeviceMountAction(
            device=x_property1.device,
            configuration=configuration,
            begin_contact=contact,
            begin_date=dateutil.parser.parse(data1["data"]["attributes"]["begin_date"]),
        )
        db.session.add(device_mount_action)
        db.session.commit()

    def test_filtered_by_x_property(self):
        """Ensure that filter by a specific device-property works."""
        data1, x_property1 = self.prepare_request_data_with_x_property(
            "test dynamic_location_begin_action1"
        )
        self.ensure_device_mount_action_exists(data1, x_property1)

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, x_property2 = self.prepare_request_data_with_x_property(
            "test dynamic_location_begin_action2"
        )
        self.ensure_device_mount_action_exists(data2, x_property2)

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        with self.client:
            response = self.client.get(
                self.url, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 2)
        # Test only for the first one
        with self.client:
            url_get_for_property1 = (
                base_url
                + f"/device-properties/{x_property1.id}/dynamic-location-actions-x"
            )
            response = self.client.get(
                url_get_for_property1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test dynamic_location_begin_action1",
        )

    def test_filtered_by_x_property_id(self):
        """Ensure that filter by filter[x_property_id]."""
        data1, x_property1 = self.prepare_request_data_with_x_property(
            "test dynamic_location_begin_action1"
        )
        self.ensure_device_mount_action_exists(data1, x_property1)

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, x_property2 = self.prepare_request_data_with_x_property(
            "test dynamic_location_begin_action2"
        )
        self.ensure_device_mount_action_exists(data2, x_property2)

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        # Test only for the first one
        with self.client:
            url_get_for_property1 = (
                base_url
                + f"/dynamic-location-actions?filter[x_property_id]={x_property1.id}"
            )
            response = self.client.get(
                url_get_for_property1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test dynamic_location_begin_action1",
        )

    def test_filtered_by_y_property_id(self):
        """Ensure that filter by filter[y_property_id]."""
        data1, y_property1 = self.prepare_request_data_with_y_property(
            "test dynamic_location_begin_action1"
        )
        self.ensure_device_mount_action_exists(data1, y_property1)

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, y_property2 = self.prepare_request_data_with_y_property(
            "test dynamic_location_begin_action2"
        )
        self.ensure_device_mount_action_exists(data2, y_property2)

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        # Test only for the first one
        with self.client:
            url_get_for_property1 = (
                base_url
                + f"/dynamic-location-actions?filter[y_property_id]={y_property1.id}"
            )
            response = self.client.get(
                url_get_for_property1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test dynamic_location_begin_action1",
        )

    def test_filtered_by_z_property_id(self):
        """Ensure that filter by filter[z_property_id]."""
        data1, z_property1 = self.prepare_request_data_with_z_property(
            "test dynamic_location_begin_action1"
        )
        self.ensure_device_mount_action_exists(data1, z_property1)

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, z_property2 = self.prepare_request_data_with_z_property(
            "test dynamic_location_begin_action2"
        )
        self.ensure_device_mount_action_exists(data2, z_property2)

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        # Test only for the first one
        with self.client:
            url_get_for_property1 = (
                base_url
                + f"/dynamic-location-actions?filter[z_property_id]={z_property1.id}"
            )
            response = self.client.get(
                url_get_for_property1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test dynamic_location_begin_action1",
        )

    def test_http_response_not_found(self):
        """Make sure that the backend responds with 404 HTTP-Code if a resource was not found."""
        url = f"{self.url}/{fake.random_int()}"
        _ = super().http_code_404_when_resource_not_found(url)

    def test_get_with_label(self):
        """Ensure that we return the label in the response."""
        dynamic_location_begin_action = add_dynamic_location_begin_action_model()
        dynamic_location_begin_action.label = "fancy location"
        configuration = dynamic_location_begin_action.configuration
        configuration.is_public = True
        configuration.is_internal = False
        db.session.add_all([dynamic_location_begin_action, configuration])
        db.session.commit()

        response = self.client.get(f"{self.url}/{dynamic_location_begin_action.id}")
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.json["data"]["attributes"]["label"], "fancy location")

    @fixtures.use(["super_user", "mount_of_public_device1_in_group1"])
    def test_post_triggers_mqtt_notification(
        self, super_user, mount_of_public_device1_in_group1
    ):
        """Ensure that we can post a dynamic location and publish the notification via mqtt."""
        payload = {
            "data": {
                "type": "configuration_dynamic_location_action",
                "attributes": {
                    "begin_date": "2022-01-24T00:00:00Z",
                },
                "relationships": {
                    "configuration": {
                        "data": {
                            "type": "configuration",
                            "id": str(
                                mount_of_public_device1_in_group1.configuration_id
                            ),
                        }
                    },
                    "begin_contact": {
                        "data": {
                            "type": "contact",
                            "id": str(
                                mount_of_public_device1_in_group1.begin_contact_id
                            ),
                        }
                    },
                },
            }
        }
        with self.run_requests_as(super_user):
            resp = self.client.post(
                self.url,
                data=json.dumps(payload),
                content_type="application/vnd.api+json",
            )
        self.expect(resp.status_code).to_equal(201)
        # And ensure that we trigger the mqtt.
        mqtt.publish.assert_called_once()
        call_args = mqtt.publish.call_args[0]

        self.expect(call_args[0]).to_equal(
            "sms/post-configuration-dynamic-location-action"
        )
        notification_data = json.loads(call_args[1])["data"]
        self.expect(notification_data["type"]).to_equal(
            "configuration_dynamic_location_action"
        )
        self.expect(notification_data["attributes"]["begin_date"]).to_equal(
            "2022-01-24T00:00:00+00:00"
        )
        self.expect(str).of(notification_data["id"]).to_match(r"\d+")

    @fixtures.use(
        ["super_user", "dynamic_location1_of_public_configuration1_in_group1"]
    )
    def test_patch_triggers_mqtt_notification(
        self,
        super_user,
        dynamic_location1_of_public_configuration1_in_group1,
    ):
        """Ensure that we can patch a dynamic location and publish the notification via mqtt."""
        with self.run_requests_as(super_user):
            resp = self.client.patch(
                f"{self.url}/{dynamic_location1_of_public_configuration1_in_group1.id}",
                data=json.dumps(
                    {
                        "data": {
                            "type": "configuration_dynamic_location_action",
                            "id": str(
                                dynamic_location1_of_public_configuration1_in_group1.id
                            ),
                            "attributes": {"end_date": "2025-01-02T00:00:00Z"},
                        }
                    }
                ),
                content_type="application/vnd.api+json",
            )
        self.expect(resp.status_code).to_equal(200)
        # And ensure that we trigger the mqtt.
        mqtt.publish.assert_called_once()
        call_args = mqtt.publish.call_args[0]

        self.expect(call_args[0]).to_equal(
            "sms/patch-configuration-dynamic-location-action"
        )
        notification_data = json.loads(call_args[1])["data"]
        self.expect(notification_data["type"]).to_equal(
            "configuration_dynamic_location_action"
        )
        self.expect(notification_data["attributes"]["end_date"]).to_equal(
            "2025-01-02T00:00:00+00:00"
        )
        self.expect(notification_data["attributes"]["begin_date"]).to_equal(
            "2022-01-25T00:00:00+00:00"
        )

    @fixtures.use(
        ["super_user", "dynamic_location1_of_public_configuration1_in_group1"]
    )
    def test_delete_triggers_mqtt_notification(
        self,
        super_user,
        dynamic_location1_of_public_configuration1_in_group1,
    ):
        """Ensure that we can delete a dynamic location and publish the notification via mqtt."""
        with self.run_requests_as(super_user):
            resp = self.client.delete(
                f"{self.url}/{dynamic_location1_of_public_configuration1_in_group1.id}",
            )
        self.expect(resp.status_code).to_equal(200)
        # And ensure that we trigger the mqtt.
        mqtt.publish.assert_called_once()
        call_args = mqtt.publish.call_args[0]

        self.expect(call_args[0]).to_equal(
            "sms/delete-configuration-dynamic-location-action"
        )
        self.expect(json.loads).of(call_args[1]).to_equal(
            {
                "data": {
                    "type": "configuration_dynamic_location_action",
                    "id": str(dynamic_location1_of_public_configuration1_in_group1.id),
                }
            }
        )

    @fixtures.use(
        ["super_user", "dynamic_location1_of_public_configuration1_in_group1"]
    )
    def test_patch_unset_xyz_properties(
        self, super_user, dynamic_location1_of_public_configuration1_in_group1
    ):
        """Ensure we can unset the z property via the api."""
        payload = {
            "data": {
                "id": str(dynamic_location1_of_public_configuration1_in_group1.id),
                "type": "configuration_dynamic_location_action",
                "relationships": {
                    "z_property": {
                        "data": None,
                    },
                },
            }
        }

        with self.run_requests_as(super_user):
            resp = self.client.patch(
                f"{self.url}/{dynamic_location1_of_public_configuration1_in_group1.id}",
                json=payload,
                headers={"Content-Type": "application/vnd.api+json"},
            )
        self.assertEqual(resp.status_code, 200)
