# SPDX-FileCopyrightText: 2021 - 2024
# - Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
# - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
# - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
# - Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)
#
# SPDX-License-Identifier: EUPL-1.2

"""Tests for the static location action api."""

import datetime
import json

import pytz

from project import base_url, db
from project.api.models import (
    Configuration,
    ConfigurationStaticLocationBeginAction,
    Contact,
    Site,
    User,
)
from project.extensions.instances import mqtt
from project.tests.base import (
    BaseTestCase,
    Fixtures,
    create_token,
    fake,
    generate_userinfo_data,
)
from project.tests.models.test_configuration_static_action_model import (
    add_static_location_begin_action_model,
)
from project.tests.models.test_configurations_model import generate_configuration_model

fixtures = Fixtures()


@fixtures.register("public_configuration1_in_group1", scope=lambda: db.session)
def create_public_configuration1_in_group1():
    """Create a public configuration that uses group 1 for permission management."""
    result = Configuration(
        label="public configuration1",
        is_internal=False,
        is_public=True,
        cfg_permission_group="1",
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register("super_user_contact", scope=lambda: db.session)
def create_super_user_contact():
    """Create a contact that can be used to make a super user."""
    result = Contact(
        given_name="super", family_name="contact", email="super.contact@localhost"
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register(
    "static_location1_of_public_configuration1_in_group1", scope=lambda: db.session
)
@fixtures.use(["public_configuration1_in_group1", "super_user_contact"])
def create_static_location1_of_public_configuration1_in_group1(
    public_configuration1_in_group1, super_user_contact
):
    """Create a static location for the public configuration."""
    result = ConfigurationStaticLocationBeginAction(
        configuration=public_configuration1_in_group1,
        begin_date=datetime.datetime(2022, 1, 25, 0, 0, 0, tzinfo=pytz.UTC),
        begin_contact=super_user_contact,
    )
    db.session.add(result)
    db.session.commit()
    return result


@fixtures.register("super_user", scope=lambda: db.session)
@fixtures.use(["super_user_contact"])
def create_super_user(super_user_contact):
    """Create super user to use it in the tests."""
    result = User(
        contact=super_user_contact, subject=super_user_contact.email, is_superuser=True
    )
    db.session.add(result)
    db.session.commit()
    return result


class TestConfigurationStaticLocationActionServices(BaseTestCase):
    """Tests for the ConfigurationStaticLocationAction endpoint."""

    url = base_url + "/static-location-actions"
    contact_url = base_url + "/contacts"
    object_type = "configuration_static_location_action"

    def test_get_configuration_static_location_action(self):
        """Ensure the List /configuration_static_location_action route behaves correctly."""
        response = self.client.get(self.url)
        self.assertEqual(response.status_code, 200)
        # no data yet
        self.assertEqual(response.json["data"], [])

    def test_get_device_static_location_action_collection(self):
        """Test retrieve a collection of configuration_static_location_action objects."""
        static_location_begin_action = add_static_location_begin_action_model(
            is_public=True, is_private=False, is_internal=False
        )
        with self.client:
            response = self.client.get(self.url)
        data = json.loads(response.data.decode())
        self.assertEqual(response.status_code, 200)
        self.assertEqual(
            static_location_begin_action.begin_description,
            data["data"][0]["attributes"]["begin_description"],
        )

    def test_add_configuration_static_begin_location_action(self):
        """
        Ensure that we can create a new static location.

        Ensure POST a new configuration static location begin action
        can be added to the database.
        """
        data, _ = self.prepare_request_data(
            "test configuration_static_location_begin_action"
        )

        result = super().add_object(
            url=self.url,
            data_object=data,
            object_type=self.object_type,
        )

        configuration_id = result["data"]["relationships"]["configuration"]["data"][
            "id"
        ]
        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        self.assertEqual(
            configuration.update_description, "create;static location action"
        )
        # But we can't add another one with the very same settings, as
        # the exact configuration has already the location for the time slot
        self.try_add_object_with_status_code(
            url=self.url,
            data_object=data,
            expected_status_code=409,  # 409 => ConflictError
        )

    def test_post_configuration_static_begin_location_action_for_archived_configuration(
        self,
    ):
        """Ensure we can't add a location to an archived configuration."""
        data, _ = self.prepare_request_data(
            "test configuration_static_location_begin_action"
        )

        configuration_id = data["data"]["relationships"]["configuration"]["data"]["id"]

        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        configuration.archived = True
        db.session.add(configuration)
        db.session.commit()

        _ = super().try_add_object_with_status_code(
            url=self.url, data_object=data, expected_status_code=403
        )

    def prepare_request_data(self, description):
        """Return a payload to create a static location action later."""
        config = generate_configuration_model(
            is_public=True, is_private=False, is_internal=False
        )
        userinfo = generate_userinfo_data()
        contact = Contact(
            given_name=userinfo["given_name"],
            family_name=userinfo["family_name"],
            email=userinfo["email"],
        )
        db.session.add_all([contact, config])
        db.session.commit()
        data = {
            "data": {
                "type": self.object_type,
                "attributes": {
                    "begin_date": "2021-10-22T10:00:50.542Z",
                    "end_date": "2026-09-22T10:00:50.542Z",
                    "begin_description": description,
                    "end_description": "end",
                    "x": str(fake.coordinate()),
                    "y": str(fake.coordinate()),
                    "z": str(fake.coordinate()),
                    "epsg_code": None,
                    "elevation_datum_name": None,
                    "elevation_datum_uri": fake.uri(),
                },
                "relationships": {
                    "begin_contact": {"data": {"type": "contact", "id": contact.id}},
                    "end_contact": {"data": {"type": "contact", "id": contact.id}},
                    "configuration": {
                        "data": {"type": "configuration", "id": config.id}
                    },
                },
            }
        }
        return data, config

    def test_update_configuration_static_begin_location_action(self):
        """Ensure a configuration_static_begin_location_action can be updated."""
        static_location_begin_action = add_static_location_begin_action_model()
        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        new_data = {
            "data": {
                "type": self.object_type,
                "id": static_location_begin_action.id,
                "attributes": {
                    "end_description": "changed",
                    "end_date": "2023-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": {"type": "contact", "id": contact["data"]["id"]}
                    },
                },
            }
        }

        result = super().update_object(
            url=f"{self.url}/{static_location_begin_action.id}",
            data_object=new_data,
            object_type=self.object_type,
        )
        configuration_id = result["data"]["relationships"]["configuration"]["data"][
            "id"
        ]
        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        self.assertEqual(
            configuration.update_description, "update;static location action"
        )

    def test_update_for_archived_configuration(self):
        """Ensure a configuration_static_begin_location_action can be updated."""
        static_location_begin_action = add_static_location_begin_action_model()

        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        new_data = {
            "data": {
                "type": self.object_type,
                "id": static_location_begin_action.id,
                "attributes": {
                    "end_description": "changed",
                    "end_date": "2023-10-22T10:00:50.542Z",
                },
                "relationships": {
                    "end_contact": {
                        "data": {"type": "contact", "id": contact["data"]["id"]}
                    },
                },
            }
        }
        static_location_begin_action.configuration.archived = True
        db.session.add(static_location_begin_action.configuration)
        db.session.commit()

        _ = super().try_update_object_with_status_code(
            url=f"{self.url}/{static_location_begin_action.id}",
            data_object=new_data,
            expected_status_code=403,
        )

    def test_update_configuration_static_begin_location_action_set_end_contact_to_none(
        self,
    ):
        """Ensure that we can reset the end_contact if necessary."""
        static_location_begin_action = add_static_location_begin_action_model()
        userinfo = generate_userinfo_data()
        contact_data = {
            "data": {
                "type": "contact",
                "attributes": {
                    "given_name": userinfo["given_name"],
                    "family_name": userinfo["family_name"],
                    "email": userinfo["email"],
                    "website": fake.url(),
                },
            }
        }
        contact_result = super().add_object(
            url=self.contact_url, data_object=contact_data, object_type="contact"
        )
        contact = (
            db.session.query(Contact).filter_by(id=contact_result["data"]["id"]).one()
        )
        static_location_begin_action.end_contact = contact
        db.session.add_all([static_location_begin_action, contact])
        db.session.commit()
        new_data = {
            "data": {
                "type": self.object_type,
                "id": static_location_begin_action.id,
                "attributes": {
                    "end_description": "changed",
                    "end_date": "2023-10-22T10:00:50.542Z",
                },
                # And we want to set the contact back to None
                "relationships": {
                    # "end_contact": {
                    #    "data": {"type": "contact", "id": None}
                    # },
                    # "end_contact": None,
                    "end_contact": {"data": None},
                },
            }
        }

        _ = super().update_object(
            url=f"{self.url}/{static_location_begin_action.id}",
            data_object=new_data,
            object_type=self.object_type,
        )

    def test_update_configuration_static_location_fail_due_to_overlapping_times_for_locations(
        self,
    ):
        """Ensure that patching fails if there is already a location for the time."""
        static_location_begin_action1 = add_static_location_begin_action_model()
        static_location_begin_action2 = add_static_location_begin_action_model()
        static_location_begin_action1.begin_date = datetime.datetime(
            2023, 1, 1, 12, 0, 0
        )
        static_location_begin_action1.end_date = datetime.datetime(
            2023, 12, 31, 12, 0, 0
        )

        static_location_begin_action2.begin_date = datetime.datetime(
            2024, 1, 1, 12, 0, 0
        )
        static_location_begin_action2.end_date = datetime.datetime(
            2024, 12, 31, 12, 0, 0
        )
        static_location_begin_action2.configuration = (
            static_location_begin_action1.configuration
        )

        db.session.add_all(
            [static_location_begin_action1, static_location_begin_action2]
        )
        db.session.commit()

        updated_data = {
            "data": {
                "type": self.object_type,
                "id": static_location_begin_action1.id,
                "attributes": {
                    "begin_date": "2023-11-11T11:11:11.000Z",
                    "end_date": "2024-08-09T00:00:50.542Z",
                },
            }
        }

        _ = super().try_update_object_with_status_code(
            url=f"{self.url}/{static_location_begin_action1.id}",
            data_object=updated_data,
            expected_status_code=409,
        )

    def test_delete_configuration_static_begin_location_action(self):
        """Ensure a configuration_static_begin_location_action can be deleted."""
        static_location_begin_action = add_static_location_begin_action_model()
        configuration_id = static_location_begin_action.configuration_id

        _ = super().delete_object(
            url=f"{self.url}/{static_location_begin_action.id}",
        )
        configuration = (
            db.session.query(Configuration).filter_by(id=configuration_id).first()
        )
        self.assertEqual(
            configuration.update_description, "delete;static location action"
        )

    def test_delete_for_archived_configuration(self):
        """Ensure we can't delete a location for an archived configuration."""
        static_location_begin_action = add_static_location_begin_action_model()
        configuration = static_location_begin_action.configuration

        configuration.archived = True
        db.session.add(configuration)
        db.session.commit()

        with self.client:
            access_headers = create_token()
            response = self.client.delete(
                f"{self.url}/{static_location_begin_action.id}",
                content_type="application/vnd.api+json",
                headers=access_headers,
            )
        self.assertEqual(response.status_code, 403)

    def test_filtered_by_configuration(self):
        """Ensure that filter by a specific configuration works well."""
        data1, config1 = self.prepare_request_data("test static_location_begin_action1")

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, config2 = self.prepare_request_data("test static_location_begin_action2")

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        with self.client:
            response = self.client.get(
                self.url, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 2)
        # Test only for the first one
        with self.client:
            url_get_for_config1 = (
                base_url + f"/configurations/{config1.id}/static-location-actions"
            )
            response = self.client.get(
                url_get_for_config1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test static_location_begin_action1",
        )

    def test_filtered_by_configuration_id(self):
        """Ensure that filter by filter[configuration_id]."""
        data1, config1 = self.prepare_request_data("test static_location_begin_action1")

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        data2, config2 = self.prepare_request_data("test static_location_begin_action2")

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        # Test only for the first one
        with self.client:
            url_get_for_config1 = (
                base_url
                + f"/static-location-actions?filter[configuration_id]={config1.id}"
            )
            response = self.client.get(
                url_get_for_config1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test static_location_begin_action1",
        )

    def test_filtered_by_site(self):
        """Ensure that filter by a specific site works well."""
        site1 = Site(label="test site 1")
        data1, config1 = self.prepare_request_data("test static_location_begin_action1")
        config1.site = site1
        db.session.add_all([config1, site1])
        db.session.commit()

        _ = super().add_object(
            url=self.url,
            data_object=data1,
            object_type=self.object_type,
        )
        site2 = Site(label="test site 2")
        data2, config2 = self.prepare_request_data("test static_location_begin_action2")
        config2.site = site2
        db.session.add_all([config2, site2])
        db.session.commit()

        _ = super().add_object(
            url=self.url,
            data_object=data2,
            object_type=self.object_type,
        )

        with self.client:
            response = self.client.get(
                self.url, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 2)
        # Test only for the first one
        with self.client:
            url_get_for_config1 = (
                base_url + f"/sites/{site1.id}/static-location-actions"
            )
            response = self.client.get(
                url_get_for_config1, content_type="application/vnd.api+json"
            )
        self.assertEqual(response.status_code, 200)
        self.assertEqual(len(response.json["data"]), 1)
        self.assertEqual(
            response.json["data"][0]["attributes"]["begin_description"],
            "test static_location_begin_action1",
        )

    def test_http_response_not_found(self):
        """Make sure that the backend responds with 404 HTTP-Code if a resource was not found."""
        url = f"{self.url}/{fake.random_int()}"
        _ = super().http_code_404_when_resource_not_found(url)

    def test_get_with_label(self):
        """Ensure that we return the label in the response."""
        static_location_begin_action = add_static_location_begin_action_model()
        static_location_begin_action.label = "fancy location"
        configuration = static_location_begin_action.configuration
        configuration.is_public = True
        configuration.is_internal = False
        db.session.add_all([static_location_begin_action, configuration])
        db.session.commit()

        response = self.client.get(f"{self.url}/{static_location_begin_action.id}")
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.json["data"]["attributes"]["label"], "fancy location")

    @fixtures.use(
        ["super_user", "super_user_contact", "public_configuration1_in_group1"]
    )
    def test_post_triggers_mqtt_notification(
        self,
        super_user,
        super_user_contact,
        public_configuration1_in_group1,
    ):
        """Ensure that we can post a static location and publish the notification via mqtt."""
        payload = {
            "data": {
                "type": "configuration_static_location_action",
                "attributes": {
                    "begin_date": "2022-01-24T00:00:00Z",
                },
                "relationships": {
                    "configuration": {
                        "data": {
                            "type": "configuration",
                            "id": str(
                                public_configuration1_in_group1.id,
                            ),
                        }
                    },
                    "begin_contact": {
                        "data": {
                            "type": "contact",
                            "id": str(
                                super_user_contact.id,
                            ),
                        }
                    },
                },
            }
        }
        with self.run_requests_as(super_user):
            resp = self.client.post(
                self.url,
                data=json.dumps(payload),
                content_type="application/vnd.api+json",
            )
        self.expect(resp.status_code).to_equal(201)
        # And ensure that we trigger the mqtt.
        mqtt.publish.assert_called_once()
        call_args = mqtt.publish.call_args[0]

        self.expect(call_args[0]).to_equal(
            "sms/post-configuration-static-location-action"
        )
        notification_data = json.loads(call_args[1])["data"]
        self.expect(notification_data["type"]).to_equal(
            "configuration_static_location_action"
        )
        self.expect(notification_data["attributes"]["begin_date"]).to_equal(
            "2022-01-24T00:00:00+00:00"
        )
        self.expect(str).of(notification_data["id"]).to_match(r"\d+")

    @fixtures.use(["super_user", "static_location1_of_public_configuration1_in_group1"])
    def test_patch_triggers_mqtt_notification(
        self,
        super_user,
        static_location1_of_public_configuration1_in_group1,
    ):
        """Ensure that we can patch a static location and publish the notification via mqtt."""
        with self.run_requests_as(super_user):
            resp = self.client.patch(
                f"{self.url}/{static_location1_of_public_configuration1_in_group1.id}",
                data=json.dumps(
                    {
                        "data": {
                            "type": "configuration_static_location_action",
                            "id": str(
                                static_location1_of_public_configuration1_in_group1.id
                            ),
                            "attributes": {"end_date": "2025-01-02T00:00:00Z"},
                        }
                    }
                ),
                content_type="application/vnd.api+json",
            )
        self.expect(resp.status_code).to_equal(200)
        # And ensure that we trigger the mqtt.
        mqtt.publish.assert_called_once()
        call_args = mqtt.publish.call_args[0]

        self.expect(call_args[0]).to_equal(
            "sms/patch-configuration-static-location-action"
        )
        notification_data = json.loads(call_args[1])["data"]
        self.expect(notification_data["type"]).to_equal(
            "configuration_static_location_action"
        )
        self.expect(notification_data["attributes"]["end_date"]).to_equal(
            "2025-01-02T00:00:00+00:00"
        )
        self.expect(notification_data["attributes"]["begin_date"]).to_equal(
            "2022-01-25T00:00:00+00:00"
        )

    @fixtures.use(["super_user", "static_location1_of_public_configuration1_in_group1"])
    def test_delete_triggers_mqtt_notification(
        self,
        super_user,
        static_location1_of_public_configuration1_in_group1,
    ):
        """Ensure that we can delete a static location and publish the notification via mqtt."""
        with self.run_requests_as(super_user):
            resp = self.client.delete(
                f"{self.url}/{static_location1_of_public_configuration1_in_group1.id}",
            )
        self.expect(resp.status_code).to_equal(200)
        # And ensure that we trigger the mqtt.
        mqtt.publish.assert_called_once()
        call_args = mqtt.publish.call_args[0]

        self.expect(call_args[0]).to_equal(
            "sms/delete-configuration-static-location-action"
        )
        self.expect(json.loads).of(call_args[1]).to_equal(
            {
                "data": {
                    "type": "configuration_static_location_action",
                    "id": str(static_location1_of_public_configuration1_in_group1.id),
                }
            }
        )
