/**
 * @license EUPL-1.2
 * SPDX-FileCopyrightText: 2020 - 2024
 * - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
 * - Marc Hanisch <marc.hanisch@gfz-potsdam.de>
 * - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
 *
 * SPDX-License-Identifier: EUPL-1.2
 */
import { Unit } from '@/models/Unit'

import {
  IJsonApiEntityListEnvelope,
  IJsonApiEntity,
  IJsonApiEntityWithoutDetails
} from '@/serializers/jsonapi/JsonApiTypes'

export class UnitSerializer {
  convertJsonApiObjectListToModelList (jsonApiObjectList: IJsonApiEntityListEnvelope): Unit[] {
    return jsonApiObjectList.data.map(this.convertJsonApiDataToModel.bind(this))
  }

  convertJsonApiDataToModel (jsonApiData: IJsonApiEntity): Unit {
    const id = jsonApiData.id.toString()
    const name = jsonApiData.attributes.term
    const url = jsonApiData.links?.self || ''
    const definition = jsonApiData.attributes.definition
    const provenance = jsonApiData.attributes.provenance || ''
    const provenanceUri = jsonApiData.attributes.provenance_uri || ''
    const provenanceTerm = jsonApiData.attributes.provenance_term || ''
    const ucumCaseSensitiveSymbol = jsonApiData.attributes.ucum_case_sensitive_symbol || ''
    const category = jsonApiData.attributes.category || ''
    const note = jsonApiData.attributes.note || ''

    let globalProvenanceId = null

    if (jsonApiData.relationships?.global_provenance?.data) {
      const data = jsonApiData.relationships.global_provenance.data as IJsonApiEntityWithoutDetails
      if (data.id) {
        globalProvenanceId = data.id
      }
    }

    return Unit.createFromObject({
      id,
      name,
      definition,
      provenance,
      provenanceUri,
      provenanceTerm,
      ucumCaseSensitiveSymbol,
      category,
      note,
      uri: url,
      globalProvenanceId
    })
  }

  convertModelToJsonApiData (unit: Unit) {
    const attributes = {
      term: unit.name,
      definition: unit.definition,
      provenance: unit.provenance,
      provenance_uri: unit.provenanceUri,
      provenance_term: unit.provenanceTerm,
      ucum_case_sensitive_symbol: unit.ucumCaseSensitiveSymbol,
      category: unit.category,
      note: unit.note
    }

    const relationships: any = {}

    if (unit.globalProvenanceId) {
      relationships.global_provenance = {
        data: {
          id: unit.globalProvenanceId,
          type: 'GlobalProvenance'
        }
      }
    }

    const wrapper: any = {
      type: 'Unit',
      attributes,
      relationships
    }

    if (unit.id) {
      wrapper.id = unit.id
    }

    return wrapper
  }
}
