/**
 * @license EUPL-1.2
 * SPDX-FileCopyrightText: 2020 - 2023
 * - Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
 * - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
 * - Marc Hanisch <marc.hanisch@gfz-potsdam.de>
 * - Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)
 * - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

import { DateTime } from 'luxon'
import { Contact } from '@/models/Contact'
import { Platform } from '@/models/Platform'
import { Attachment } from '@/models/Attachment'
import { Parameter } from '@/models/Parameter'

import { PlatformSerializer, IPlatformWithMeta, platformWithMetaToPlatformByThrowingErrorOnMissing, platformWithMetaToPlatformByAddingDummyObjects } from '@/serializers/jsonapi/PlatformSerializer'
import { IJsonApiEntityWithoutDetailsDataDictList, IJsonApiEntity } from '@/serializers/jsonapi/JsonApiTypes'

const createTestPlatform = () => {
  const platform = new Platform()
  platform.description = 'This is a dummy description'
  platform.shortName = 'Dummy short name'
  platform.longName = 'Dummy long long long name'
  platform.serialNumber = '12345'
  platform.inventoryNumber = '6789'
  platform.manufacturerUri = 'manufacturer/manu1'
  platform.manufacturerName = 'Manu1'
  platform.platformTypeUri = 'platformType/typeA'
  platform.platformTypeName = 'Type A'
  platform.statusUri = 'status/Ok'
  platform.statusName = 'Okay'
  platform.model = '0815'
  platform.persistentIdentifier = 'doi:4354545'
  platform.website = 'http://gfz-potsdam.de'
  platform.createdAt = DateTime.utc(2020, 8, 28, 13, 49, 48, 15)
  platform.updatedAt = DateTime.utc(2020, 8, 30, 13, 49, 48, 15)
  platform.updateDescription = ''
  platform.keywords = ['key', 'word']
  platform.country = 'Germany'

  platform.attachments = [
    Attachment.createFromObject({
      id: '2',
      label: 'GFZ',
      url: 'http://www.gfz-potsdam.de',
      description: 'The GFZ homepage',
      isUpload: false,
      createdAt: null
    }),
    Attachment.createFromObject({
      id: null,
      label: 'UFZ',
      url: 'http://www.ufz.de',
      description: 'The UFZ homepage',
      isUpload: false,
      createdAt: null
    })
  ]

  platform.parameters = [
    Parameter.createFromObject({
      id: '3',
      label: 'Parameter 1',
      description: 'Some paramter',
      unitUri: 'unit/Unit1',
      unitName: 'Unit 1'
    }),
    Parameter.createFromObject({
      id: '4',
      label: 'Parameter 2',
      description: 'Some other paramter',
      unitUri: 'unit/Unit2',
      unitName: 'Unit 2'
    })
  ]

  platform.contacts = [
    Contact.createFromObject({
      id: '4',
      givenName: 'Max',
      familyName: 'Mustermann',
      email: 'max@mustermann.de',
      website: '',
      organization: '',
      orcid: '',
      createdByUserId: null,
      createdAt: null,
      updatedAt: null
    }),
    Contact.createFromObject({
      id: '5',
      givenName: 'Mux',
      familyName: 'Mastermunn',
      email: 'mux@mastermunn.de',
      website: '',
      organization: '',
      orcid: '',
      createdByUserId: null,
      createdAt: null,
      updatedAt: null
    })
  ]
  return platform
}

describe('PlatformSerializer', () => {
  describe('#convertJsonApiObjectListToModelList', () => {
    it('should convert a json api object with multiple entries to a platform model list', () => {
      const jsonApiObjectList: any = {
        data: [{
          type: 'platform',
          attributes: {
            serial_number: '000123',
            model: '0815',
            description: 'Soil Moisture station Boeken_BF1',
            attachments: [{
              label: 'GFZ',
              url: 'http://www.gfz-potsdam.de',
              id: '12'
            }],
            platform_type_uri: 'type/Station',
            status_uri: 'status/inuse',
            website: 'http://www.tereno.net',
            long_name: 'Soil moisture station Boeken BF1, Germany',
            inventory_number: '0001234',
            manufacturer_name: 'XYZ',
            short_name: 'boeken_BF1',
            status_name: 'in use',
            platform_type_name: 'Station',
            persistent_identifier: 'boeken_BF1',
            manufacturer_uri: 'manufacturer/xyz',
            archived: true,
            keywords: ['key', 'word'],
            country: 'Germany'
          },
          relationships: {
            updated_by: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/updatedUser'
              }
            },
            contacts: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/contacts',
                related: '/rdm/svm-api/v1/platforms/37/contacts'
              },
              data: [{
                type: 'contact',
                id: '1'
              }]
            },
            platform_attachments: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/platform-attachments',
                related: '/rdm/svm-api/v1/platforms/37/platform-attachments'
              },
              data: [{
                type: 'platform_attachment',
                id: '12'
              }]
            },
            platform_parameters: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/platform-parameters',
                related: '/rdm/svm-api/v1/platforms/37/platform-parameters'
              },
              data: [{
                type: 'platform_parameter',
                id: '1'
              }]
            },
            created_by: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/createdUser'
              },
              data: {
                id: '42',
                type: 'user'
              }
            }
          },
          id: '37',
          links: {
            self: '/rdm/svm-api/v1/platforms/37'
          }
        }, {
          type: 'platform',
          attributes: {
            serial_number: null,
            model: null,
            description: 'Groundwater level KleinTrebbow',
            platform_type_uri: 'Station',
            status_uri: null,
            website: null,
            long_name: 'Groundwater level KleinTrebbow',
            inventory_number: null,
            manufacturer_name: null,
            short_name: 'klein_trebbow',
            status_name: null,
            platform_type_name: 'Station',
            persistent_identifier: null,
            manufacturer_uri: null,
            archived: false,
            country: null
          },
          relationships: {
            updated_by: {
              links: {
                self: '/rdm/svm-api/v1/platforms/52/relationships/updatedUser'
              }
            },
            contacts: {
              links: {
                self: '/rdm/svm-api/v1/platforms/52/relationships/contacts',
                related: '/rdm/svm-api/v1/platforms/52/contacts'
              },
              data: []
            },
            platform_attachments: {
              links: {
                self: '/rdm/svm-api/v1/platforms/52/relationships/platform-attachments',
                related: '/rdm/svm-api/v1/platforms/52/platform-attachments'
              },
              data: []
            },
            platform_parameters: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/platform-parameters',
                related: '/rdm/svm-api/v1/platforms/37/platform-parameters'
              },
              data: [{
                type: 'platform_parameter',
                id: '2'
              }]
            },
            created_by: {
              links: {
                self: '/rdm/svm-api/v1/platforms/52/relationships/createdUser'
              }
            }
          },
          id: '52',
          links: {
            self: '/rdm/svm-api/v1/platforms/52'
          }
        }],
        links: {
          self: 'http://rz-vm64.gfz-potsdam.de:5000/rdm/svm-api/v1/platforms?include=contacts&filter=%5B%7B%22name%22%3A%22short_name%22%2C%22op%22%3A%22ilike%22%2C%22val%22%3A%22%25bo%25%25%22%7D%5D'
        },
        included: [
          {
            type: 'contact',
            relationships: {
              configurations: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/configurations',
                  related: '/rdm/svm-api/v1/configurations?contact_id=1'
                }
              },
              user: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/user',
                  related: '/rdm/svm-api/v1/contacts/1/users'
                },
                data: {
                  type: 'user',
                  id: '[<User 1>]'
                }
              },
              devices: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/devices',
                  related: '/rdm/svm-api/v1/devices?contact_id=1'
                }
              },
              platforms: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/platforms',
                  related: '/rdm/svm-api/v1/contacts/1/platforms'
                }
              }
            },
            attributes: {
              given_name: 'Max',
              email: 'test@test.test',
              website: null,
              family_name: 'Mustermann'
            },
            id: '1',
            links: {
              self: '/rdm/svm-api/v1/contacts/1'
            }
          },
          {
            type: 'platform_attachment',
            attributes: {
              url: 'http://test.test',
              label: 'test label',
              description: 'test'
            },
            relationships: {
              platform: {
                links: {
                  self: '/rdm/svm-api/v1/platform-attachments/1/relationships/platform',
                  related: '/rdm/svm-api/v1/platforms/52'
                },
                data: {
                  type: 'platform',
                  id: '52'
                }
              }
            },
            id: '12',
            links: {
              self: '/rdm/svm-api/v1/platform-attachments/1'
            }
          },
          {
            type: 'platform_parameter',
            attributes: {
              label: 'parameter a',
              description: 'some test value for parameter a',
              unit_uri: '/units/1',
              unit_name: 'unit 1'
            },
            relationships: {
              platform: {
                links: {
                  self: '/rdm/svm-api/v1/platform-parameters/1/relationships/platform',
                  related: '/rdm/svm-api/v1/platforms/52'
                },
                data: {
                  type: 'platform',
                  id: '52'
                }
              }
            },
            id: '1',
            links: {
              self: '/rdm/svm-api/v1/platform-parameters/1'
            }
          },
          {
            type: 'platform_parameter',
            attributes: {
              label: 'parameter b',
              description: 'some test value for parameter b',
              unit_uri: '/units/2',
              unit_name: 'unit 2'
            },
            relationships: {
              platform: {
                links: {
                  self: '/rdm/svm-api/v1/platform-parameters/1/relationships/platform',
                  related: '/rdm/svm-api/v1/platforms/52'
                },
                data: {
                  type: 'platform',
                  id: '52'
                }
              }
            },
            id: '2',
            links: {
              self: '/rdm/svm-api/v1/platform-parameters/2'
            }
          }
        ],
        meta: {
          count: 2
        },
        jsonapi: {
          version: '1.0'
        }
      }

      const expectedPlatform1 = new Platform()
      expectedPlatform1.id = '37'
      expectedPlatform1.serialNumber = '000123'
      expectedPlatform1.model = '0815'
      expectedPlatform1.description = 'Soil Moisture station Boeken_BF1'
      expectedPlatform1.attachments = [Attachment.createFromObject({
        id: '12',
        label: 'test label',
        url: 'http://test.test',
        description: 'test',
        isUpload: false,
        createdAt: null
      })]
      expectedPlatform1.parameters = [Parameter.createFromObject({
        id: '1',
        label: 'parameter a',
        description: 'some test value for parameter a',
        unitUri: '/units/1',
        unitName: 'unit 1'
      })]
      expectedPlatform1.platformTypeUri = 'type/Station'
      expectedPlatform1.statusUri = 'status/inuse'
      expectedPlatform1.website = 'http://www.tereno.net'
      // expectedPlatform1.updatedAt = DateTime.utc(2020, 8, 29, 13, 48, 35, 740)
      expectedPlatform1.longName = 'Soil moisture station Boeken BF1, Germany'
      // expectedPlatform1.createdAt = DateTime.utc(2020, 8, 28, 13, 48, 35, 740)
      expectedPlatform1.inventoryNumber = '0001234'
      expectedPlatform1.manufacturerName = 'XYZ'
      expectedPlatform1.shortName = 'boeken_BF1'
      expectedPlatform1.statusName = 'in use'
      expectedPlatform1.platformTypeName = 'Station'
      expectedPlatform1.persistentIdentifier = 'boeken_BF1'
      expectedPlatform1.manufacturerUri = 'manufacturer/xyz'
      expectedPlatform1.country = 'Germany'
      expectedPlatform1.contacts = [Contact.createFromObject({
        id: '1',
        givenName: 'Max',
        email: 'test@test.test',
        website: '',
        organization: '',
        familyName: 'Mustermann',
        orcid: '',
        createdByUserId: null,
        createdAt: null,
        updatedAt: null
      })]
      expectedPlatform1.createdByUserId = '42'
      expectedPlatform1.archived = true
      expectedPlatform1.keywords = ['key', 'word']

      const expectedPlatform2 = new Platform()
      expectedPlatform2.id = '52'
      expectedPlatform2.serialNumber = ''
      expectedPlatform2.model = ''
      expectedPlatform2.description = 'Groundwater level KleinTrebbow'
      expectedPlatform2.attachments = []
      expectedPlatform2.parameters = [Parameter.createFromObject({
        id: '2',
        label: 'parameter b',
        description: 'some test value for parameter b',
        unitUri: '/units/2',
        unitName: 'unit 2'
      })]
      expectedPlatform2.platformTypeUri = 'Station'
      expectedPlatform2.statusUri = ''
      expectedPlatform2.website = ''
      expectedPlatform2.updatedAt = null
      expectedPlatform2.longName = 'Groundwater level KleinTrebbow'
      expectedPlatform2.createdAt = null
      expectedPlatform2.inventoryNumber = ''
      expectedPlatform2.manufacturerName = ''
      expectedPlatform2.shortName = 'klein_trebbow'
      expectedPlatform2.statusName = ''
      expectedPlatform2.platformTypeName = 'Station'
      expectedPlatform2.persistentIdentifier = ''
      expectedPlatform2.manufacturerUri = ''
      expectedPlatform2.contacts = []
      expectedPlatform2.archived = false
      expectedPlatform2.country = ''

      const serializer = new PlatformSerializer()

      const platformsWithMeta = serializer.convertJsonApiObjectListToModelList(jsonApiObjectList)
      const platforms = platformsWithMeta.map((x: IPlatformWithMeta) => x.platform)
      expect(Array.isArray(platforms)).toBeTruthy()
      expect(platforms.length).toEqual(2)

      expect(platforms[0]).toEqual(expectedPlatform1)
      expect(platforms[1]).toEqual(expectedPlatform2)

      const missingContactIds = platformsWithMeta.map((x: IPlatformWithMeta) => {
        return x.missing.contacts.ids
      })

      expect(Array.isArray(missingContactIds)).toBeTruthy()
      expect(missingContactIds.length).toEqual(2)

      expect(missingContactIds[0]).toEqual([])
      expect(missingContactIds[1]).toEqual([])
    })
  })
  describe('#convertJsonApiObjectToModel', () => {
    it('should convert a json api objet to a platform model', () => {
      const jsonApiObject: any = {
        data: {
          type: 'platform',
          attributes: {
            serial_number: '000123',
            model: '0815',
            description: 'Soil Moisture station Boeken_BF1',
            platform_type_uri: 'type/Station',
            status_uri: 'status/inuse',
            website: 'http://www.tereno.net',
            updated_at: '2020-08-28T13:48:35.740944+00:00',
            long_name: 'Soil moisture station Boeken BF1, Germany',
            created_at: '2020-08-28T13:48:35.740944+00:00',
            inventory_number: '0001234',
            manufacturer_name: 'XYZ',
            short_name: 'boeken_BF1',
            status_name: 'in use',
            platform_type_name: 'Station',
            persistent_identifier: 'boeken_BF1',
            manufacturer_uri: 'manufacturer/xyz',
            archived: true
          },
          relationships: {
            updated_by: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/updatedUser'
              }
            },
            contacts: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/contacts',
                related: '/rdm/svm-api/v1/platforms/37/contacts'
              },
              data: [{
                type: 'contact',
                id: '1'
              }]
            },
            platform_attachments: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/platform-attachments',
                related: '/rdm/svm-api/v1/platforms/37/platform-attachments'
              },
              data: [{
                type: 'platform_attachment',
                id: '12'
              }]
            },
            platform_parameters: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/platform-parameters',
                related: '/rdm/svm-api/v1/platforms/37/platform-parameters'
              },
              data: [{
                type: 'platform_parameter',
                id: '1'
              }]
            },
            created_by: {
              links: {
                self: '/rdm/svm-api/v1/platforms/37/relationships/createdUser'
              }
            }
          },
          id: '37',
          links: {
            self: '/rdm/svm-api/v1/platforms/37'
          }
        },
        included: [
          {
            type: 'contact',
            relationships: {
              configurations: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/configurations',
                  related: '/rdm/svm-api/v1/configurations?contact_id=1'
                }
              },
              user: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/user',
                  related: '/rdm/svm-api/v1/contacts/1/users'
                },
                data: {
                  type: 'user',
                  id: '[<User 1>]'
                }
              },
              devices: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/devices',
                  related: '/rdm/svm-api/v1/devices?contact_id=1'
                }
              },
              platforms: {
                links: {
                  self: '/rdm/svm-api/v1/contacts/1/relationships/platforms',
                  related: '/rdm/svm-api/v1/contacts/1/platforms'
                }
              }
            },
            attributes: {
              given_name: 'Max',
              email: 'test@test.test',
              website: null,
              family_name: 'Mustermann'
            },
            id: '1',
            links: {
              self: '/rdm/svm-api/v1/contacts/1'
            }
          },
          {
            type: 'platform_attachment',
            attributes: {
              url: 'http://www.gfz-potsdam.de',
              label: 'GFZ',
              description: 'The GFZ homepage'
            },
            relationships: {
              platform: {
                links: {
                  self: '/rdm/svm-api/v1/platform-attachments/1/relationships/platform',
                  related: '/rdm/svm-api/v1/platforms/52'
                },
                data: {
                  type: 'platform',
                  id: '52'
                }
              }
            },
            id: '12',
            links: {
              self: '/rdm/svm-api/v1/platform-attachments/1'
            }
          },
          {
            type: 'platform_parameter',
            attributes: {
              label: 'parameter a',
              description: 'some test value for parameter a',
              unit_uri: '/units/1',
              unit_name: 'unit 1'
            },
            relationships: {
              platform: {
                links: {
                  self: '/rdm/svm-api/v1/platform-parameters/1/relationships/platform',
                  related: '/rdm/svm-api/v1/platforms/37'
                },
                data: {
                  type: 'platform',
                  id: '37'
                }
              }
            },
            id: '1',
            links: {
              self: '/rdm/svm-api/v1/platform-parameters/1'
            }
          }
        ],
        meta: {
          count: 2
        },
        jsonapi: {
          version: '1.0'
        }
      }

      const expectedPlatform = new Platform()
      expectedPlatform.id = '37'
      expectedPlatform.serialNumber = '000123'
      expectedPlatform.model = '0815'
      expectedPlatform.description = 'Soil Moisture station Boeken_BF1'
      expectedPlatform.attachments = [Attachment.createFromObject({
        id: '12',
        label: 'GFZ',
        url: 'http://www.gfz-potsdam.de',
        description: 'The GFZ homepage',
        isUpload: false,
        createdAt: null
      })]
      expectedPlatform.parameters = [Parameter.createFromObject({
        id: '1',
        label: 'parameter a',
        description: 'some test value for parameter a',
        unitUri: '/units/1',
        unitName: 'unit 1'
      })]
      expectedPlatform.platformTypeUri = 'type/Station'
      expectedPlatform.statusUri = 'status/inuse'
      expectedPlatform.website = 'http://www.tereno.net'
      expectedPlatform.updatedAt = DateTime.utc(2020, 8, 28, 13, 48, 35, 740)
      expectedPlatform.longName = 'Soil moisture station Boeken BF1, Germany'
      expectedPlatform.createdAt = DateTime.utc(2020, 8, 28, 13, 48, 35, 740)
      expectedPlatform.inventoryNumber = '0001234'
      expectedPlatform.manufacturerName = 'XYZ'
      expectedPlatform.shortName = 'boeken_BF1'
      expectedPlatform.statusName = 'in use'
      expectedPlatform.platformTypeName = 'Station'
      expectedPlatform.persistentIdentifier = 'boeken_BF1'
      expectedPlatform.manufacturerUri = 'manufacturer/xyz'
      expectedPlatform.contacts = [Contact.createFromObject({
        id: '1',
        givenName: 'Max',
        email: 'test@test.test',
        website: '',
        organization: '',
        familyName: 'Mustermann',
        orcid: '',
        createdByUserId: null,
        createdAt: null,
        updatedAt: null
      })]
      expectedPlatform.archived = true

      const serializer = new PlatformSerializer()

      const platformWithMeta = serializer.convertJsonApiObjectToModel(jsonApiObject)
      const platform = platformWithMeta.platform

      expect(platform).toEqual(expectedPlatform)

      const missingContactIds = platformWithMeta.missing.contacts.ids
      expect(Array.isArray(missingContactIds)).toBeTruthy()
      expect(missingContactIds.length).toEqual(0)
    })
  })
  describe('#convertJsonApiDataToModel', () => {
    it('should convert a json api data to a platform model', () => {
      const jsonApiData: any = {
        type: 'platform',
        attributes: {
          serial_number: '000123',
          model: '0815',
          description: 'Soil Moisture station Boeken_BF1',
          platform_type_uri: 'type/Station',
          status_uri: 'status/inuse',
          website: 'http://www.tereno.net',
          updated_at: '2020-08-28T13:48:35.740944+00:00',
          long_name: 'Soil moisture station Boeken BF1, Germany',
          created_at: '2020-08-28T13:48:35.740944+00:00',
          inventory_number: '0001234',
          manufacturer_name: 'XYZ',
          short_name: 'boeken_BF1',
          status_name: 'in use',
          platform_type_name: 'Station',
          persistent_identifier: 'boeken_BF1',
          manufacturer_uri: 'manufacturer/xyz',
          archived: false
        },
        relationships: {
          updated_by: {
            links: {
              self: '/rdm/svm-api/v1/platforms/37/relationships/updatedUser'
            }
          },
          contacts: {
            links: {
              self: '/rdm/svm-api/v1/platforms/37/relationships/contacts',
              related: '/rdm/svm-api/v1/platforms/37/contacts'
            },
            data: [{
              type: 'contact',
              id: '1'
            }]
          },
          platform_attachments: {
            links: {
              self: '/rdm/svm-api/v1/platforms/37/relationships/platform-attachments',
              related: '/rdm/svm-api/v1/platforms/37/platform-attachments'
            },
            data: [
              {
                type: 'platform_attachment',
                id: '12'
              }
            ]
          },
          platform_parameters: {
            links: {
              self: '/rdm/svm-api/v1/platforms/37/relationships/platform-parameters',
              related: '/rdm/svm-api/v1/platforms/37/platform-parameters'
            },
            data: [
              {
                type: 'platform_parameter',
                id: '1'
              }
            ]
          },
          created_by: {
            links: {
              self: '/rdm/svm-api/v1/platforms/37/relationships/createdUser'
            }
          }
        },
        id: '37',
        links: {
          self: '/rdm/svm-api/v1/platforms/37'
        }
      }
      const included: IJsonApiEntity[] = [
        {
          type: 'contact',
          relationships: {
            configurations: {
              links: {
                self: '/rdm/svm-api/v1/contacts/1/relationships/configurations',
                related: '/rdm/svm-api/v1/configurations?contact_id=1'
              },
              data: []
            },
            user: {
              links: {
                self: '/rdm/svm-api/v1/contacts/1/relationships/user',
                related: '/rdm/svm-api/v1/contacts/1/users'
              },
              data: {
                type: 'user',
                id: '[<User 1>]'
              }
            },
            devices: {
              links: {
                self: '/rdm/svm-api/v1/contacts/1/relationships/devices',
                related: '/rdm/svm-api/v1/devices?contact_id=1'
              },
              data: []
            },
            platforms: {
              links: {
                self: '/rdm/svm-api/v1/contacts/1/relationships/platforms',
                related: '/rdm/svm-api/v1/contacts/1/platforms'
              },
              data: []
            }
          },
          attributes: {
            given_name: 'Max',
            email: 'test@test.test',
            website: null,
            family_name: 'Mustermann'
          },
          id: '1',
          links: {
            self: '/rdm/svm-api/v1/contacts/1'
          }
        },
        {
          type: 'platform_attachment',
          relationships: {
            platforms: {
              links: {
                self: '/rdm/svm-api/v1/platform-attachments/1/relationships/platform',
                related: '/rdm/svm-api/v1/platforms/37'
              },
              data: {
                id: '37',
                type: 'platform'
              }
            }
          },
          attributes: {
            url: 'http://www.gfz-potsdam.de',
            label: 'GFZ',
            description: 'The GFZ homepage'
          },
          id: '12',
          links: {
            self: '/rdm/svm-api/v1/platform-attachments/1'
          }
        },
        {
          type: 'platform_parameter',
          attributes: {
            label: 'parameter a',
            description: 'some test value for parameter a',
            unit_uri: '/units/1',
            unit_name: 'unit 1'
          },
          relationships: {
            platform: {
              links: {
                self: '/rdm/svm-api/v1/platform-parameters/1/relationships/platform',
                related: '/rdm/svm-api/v1/platforms/37'
              },
              data: {
                type: 'platform',
                id: '37'
              }
            }
          },
          id: '1',
          links: {
            self: '/rdm/svm-api/v1/platform-parameters/1'
          }
        }
      ]

      const expectedPlatform = new Platform()
      expectedPlatform.id = '37'
      expectedPlatform.serialNumber = '000123'
      expectedPlatform.model = '0815'
      expectedPlatform.description = 'Soil Moisture station Boeken_BF1'
      expectedPlatform.attachments = [Attachment.createFromObject({
        id: '12',
        label: 'GFZ',
        url: 'http://www.gfz-potsdam.de',
        description: 'The GFZ homepage',
        isUpload: false,
        createdAt: null
      })]
      expectedPlatform.parameters = [Parameter.createFromObject({
        id: '1',
        label: 'parameter a',
        description: 'some test value for parameter a',
        unitUri: '/units/1',
        unitName: 'unit 1'
      })]
      expectedPlatform.platformTypeUri = 'type/Station'
      expectedPlatform.statusUri = 'status/inuse'
      expectedPlatform.website = 'http://www.tereno.net'
      expectedPlatform.updatedAt = DateTime.utc(2020, 8, 28, 13, 48, 35, 740)
      expectedPlatform.longName = 'Soil moisture station Boeken BF1, Germany'
      expectedPlatform.createdAt = DateTime.utc(2020, 8, 28, 13, 48, 35, 740)
      expectedPlatform.inventoryNumber = '0001234'
      expectedPlatform.manufacturerName = 'XYZ'
      expectedPlatform.shortName = 'boeken_BF1'
      expectedPlatform.statusName = 'in use'
      expectedPlatform.platformTypeName = 'Station'
      expectedPlatform.persistentIdentifier = 'boeken_BF1'
      expectedPlatform.manufacturerUri = 'manufacturer/xyz'
      expectedPlatform.contacts = [Contact.createFromObject({
        id: '1',
        givenName: 'Max',
        email: 'test@test.test',
        website: '',
        organization: '',
        familyName: 'Mustermann',
        orcid: '',
        createdByUserId: null,
        createdAt: null,
        updatedAt: null
      })]
      expectedPlatform.archived = false

      const serializer = new PlatformSerializer()
      const platfromWithMeta = serializer.convertJsonApiDataToModel(jsonApiData, included)
      const platform = platfromWithMeta.platform

      expect(platform).toEqual(expectedPlatform)

      const missingContactIds = platfromWithMeta.missing.contacts.ids
      expect(Array.isArray(missingContactIds)).toBeTruthy()
      expect(missingContactIds.length).toEqual(0)
    })
  })
  describe('#convertModelToJsonApiData', () => {
    it('should convert a model to json data object with all of the subelements', () => {
      const platform = createTestPlatform()

      const serializer = new PlatformSerializer()

      const includeRelationships = true
      const jsonApiData = serializer.convertModelToJsonApiData(platform, includeRelationships)

      expect(typeof jsonApiData).toEqual('object')

      expect(jsonApiData).not.toHaveProperty('id')

      expect(jsonApiData).toHaveProperty('type')
      expect(jsonApiData.type).toEqual('platform')

      expect(jsonApiData).toHaveProperty('attributes')
      const attributes = jsonApiData.attributes
      expect(typeof attributes).toBe('object')

      expect(attributes).toHaveProperty('description')
      expect(attributes.description).toEqual('This is a dummy description')
      expect(attributes).toHaveProperty('short_name')
      expect(attributes.short_name).toEqual('Dummy short name')
      expect(attributes).toHaveProperty('long_name')
      expect(attributes.long_name).toEqual('Dummy long long long name')
      expect(attributes).toHaveProperty('serial_number')
      expect(attributes.serial_number).toEqual('12345')
      expect(attributes).toHaveProperty('inventory_number')
      expect(attributes.inventory_number).toEqual('6789')
      expect(attributes).toHaveProperty('manufacturer_uri')
      expect(attributes.manufacturer_uri).toEqual('manufacturer/manu1')
      expect(attributes).toHaveProperty('manufacturer_name')
      expect(attributes.manufacturer_name).toEqual('Manu1')
      expect(attributes).toHaveProperty('platform_type_uri')
      expect(attributes.platform_type_uri).toEqual('platformType/typeA')
      expect(attributes).toHaveProperty('platform_type_name')
      expect(attributes.platform_type_name).toEqual('Type A')
      expect(attributes).toHaveProperty('status_uri')
      expect(attributes.status_uri).toEqual('status/Ok')
      expect(attributes).toHaveProperty('status_name')
      expect(attributes.status_name).toEqual('Okay')
      expect(attributes).toHaveProperty('model')
      expect(attributes.model).toEqual('0815')
      expect(attributes).toHaveProperty('persistent_identifier')
      expect(attributes.persistent_identifier).toEqual('doi:4354545')
      expect(attributes).toHaveProperty('website')
      expect(attributes.website).toEqual('http://gfz-potsdam.de')
      expect(attributes).toHaveProperty('keywords')
      expect(attributes.keywords).toEqual(['key', 'word'])
      expect(attributes).toHaveProperty('country')
      expect(attributes.country).toEqual('Germany')
      // expect(attributes).toHaveProperty('created_at')
      // expect(attributes.created_at).toEqual('2020-08-28T13:49:48.015620+00:00')
      // I wasn't able to find the exact date time format, so we use ISO date times
      // expect(attributes.created_at).toEqual('2020-08-28T13:49:48.015Z')
      // expect(attributes).toHaveProperty('updated_at')
      // expect(attributes.updated_at).toEqual('2020-08-30T13:49:48.015620+00:00')
      // again, iso date times
      // expect(attributes.updated_at).toEqual('2020-08-30T13:49:48.015Z')

      expect(jsonApiData.relationships).toHaveProperty('platform_attachments')
      const attachments = jsonApiData.relationships?.platform_attachments as IJsonApiEntityWithoutDetailsDataDictList
      expect(attachments).toHaveProperty('data')
      const attachmentData = attachments.data
      expect(Array.isArray(attachmentData)).toBeTruthy()
      expect(attachmentData.length).toEqual(1)
      expect(attachmentData[0]).toEqual({
        id: '2',
        type: 'platform_attachment'
      })

      expect(jsonApiData.relationships).toHaveProperty('platform_parameters')
      const parameters = jsonApiData.relationships?.platform_parameters as IJsonApiEntityWithoutDetailsDataDictList
      expect(parameters).toHaveProperty('data')
      const parameterData = parameters.data
      expect(Array.isArray(parameterData)).toBeTruthy()
      expect(parameterData.length).toEqual(2)
      expect(parameterData[0]).toEqual({
        id: '3',
        type: 'platform_parameter'
      })
      expect(parameterData[1]).toEqual({
        id: '4',
        type: 'platform_parameter'
      })

      expect(jsonApiData).toHaveProperty('relationships')
      expect(typeof jsonApiData.relationships).toEqual('object')
      expect(jsonApiData.relationships).toHaveProperty('contacts')
      expect(typeof jsonApiData.relationships?.contacts).toBe('object')
      // we test for the inner structure of the result anyway
      // this cast is just to tell typescript that
      // we have an array of data, so that it doesn't show
      // typeerrors here
      const contactObject = jsonApiData.relationships?.contacts as IJsonApiEntityWithoutDetailsDataDictList
      expect(contactObject).toHaveProperty('data')
      const contactData = contactObject.data
      expect(Array.isArray(contactData)).toBeTruthy()
      expect(contactData.length).toEqual(2)
      expect(contactData[0]).toEqual({
        id: '4',
        type: 'contact'
      })
      expect(contactData[1]).toEqual({
        id: '5',
        type: 'contact'
      })
    })
    it('should serialize an empty string as persistent identifier as null', () => {
      const platform = createTestPlatform()
      platform.persistentIdentifier = ''

      const serializer = new PlatformSerializer()

      const includeRelationships = true
      const jsonApiData = serializer.convertModelToJsonApiData(platform, includeRelationships)

      expect(typeof jsonApiData).toEqual('object')
      expect(jsonApiData).toHaveProperty('attributes')
      const attributes = jsonApiData.attributes
      expect(typeof attributes).toBe('object')
      expect(attributes).toHaveProperty('persistent_identifier')
      expect(attributes.persistent_identifier).toBeNull()
    })
    it('should set an id if given for the platform', () => {
      const platform = createTestPlatform()
      platform.id = 'abc'

      const serializer = new PlatformSerializer()

      const includeRelationships = false
      const jsonApiData = serializer.convertModelToJsonApiData(platform, includeRelationships)

      expect(typeof jsonApiData).toEqual('object')
      expect(jsonApiData).toHaveProperty('id')
      expect(jsonApiData.id).toEqual('abc')
    })
    it('should not include any relationship if we ask it not to do so', () => {
      const platform = createTestPlatform()
      platform.id = 'abc'

      const attachment = Attachment.createFromObject({
        id: '1',
        label: 'GFZ',
        url: 'https://www.gfz-potsdam.de',
        description: 'The GFZ homepage',
        isUpload: false,
        createdAt: null
      })

      platform.attachments = [attachment]

      const serializer = new PlatformSerializer()
      const includeRelationships = false

      const jsonApiData = serializer.convertModelToJsonApiData(platform, includeRelationships)

      expect(typeof jsonApiData).toEqual('object')
      expect(jsonApiData).toHaveProperty('id')
      expect(jsonApiData.id).toEqual('abc')

      expect(jsonApiData).not.toHaveProperty('relationships')
    })
    it('should include relationship if we want it', () => {
      const platform = createTestPlatform()
      platform.id = 'abc'

      const attachment = Attachment.createFromObject({
        id: '1',
        label: 'GFZ',
        url: 'https://www.gfz-potsdam.de',
        description: 'The GFZ homepage',
        isUpload: false,
        createdAt: null
      })

      platform.attachments = [attachment]

      const serializer = new PlatformSerializer()
      const includeRelationships = true

      const jsonApiData = serializer.convertModelToJsonApiData(platform, includeRelationships)

      expect(typeof jsonApiData).toEqual('object')
      expect(jsonApiData).toHaveProperty('id')
      expect(jsonApiData.id).toEqual('abc')

      expect(jsonApiData).toHaveProperty('relationships')
      expect(jsonApiData.relationships).toHaveProperty('platform_attachments')
    })
  })
})
describe('platformWithMetaToPlatformByThrowingErrorOnMissing', () => {
  it('should work without missing data', () => {
    const platform = new Platform()
    const missing = {
      contacts: {
        ids: []
      }
    }

    const result = platformWithMetaToPlatformByThrowingErrorOnMissing({
      platform,
      missing
    })

    expect(result).toEqual(platform)
    expect(result.contacts).toEqual([])
  })
  it('should also work if there is an contact', () => {
    const platform = new Platform()
    const contact = Contact.createFromObject({
      id: '1',
      familyName: 'Mustermann',
      givenName: 'Max',
      website: '',
      organization: '',
      email: 'max@mustermann.de',
      orcid: '',
      createdByUserId: null,
      createdAt: null,
      updatedAt: null
    })
    platform.contacts.push(contact)

    const missing = {
      contacts: {
        ids: []
      }
    }

    const result = platformWithMetaToPlatformByThrowingErrorOnMissing({
      platform,
      missing
    })

    expect(result).toEqual(platform)
    expect(result.contacts).toEqual([contact])
  })
  it('should throw an error if there are missing data', () => {
    const platform = new Platform()
    const missing = {
      contacts: {
        ids: ['1']
      }
    }

    try {
      platformWithMetaToPlatformByThrowingErrorOnMissing({
        platform,
        missing
      })
      fail('There must be an error')
    } catch (error: any) {
      expect(error.toString()).toMatch(/Contacts are missing/)
    }
  })
})
describe('platformWithMetaToPlatformByAddingDummyObjects', () => {
  it('should leave the data as it is if there are no missing data', () => {
    const platform = new Platform()
    const missing = {
      contacts: {
        ids: []
      }
    }

    const result = platformWithMetaToPlatformByAddingDummyObjects({
      platform,
      missing
    })

    expect(result).toEqual(platform)
    expect(result.contacts).toEqual([])
  })
  it('should stay with existing contacts without adding dummy data', () => {
    const platform = new Platform()
    const contact = Contact.createFromObject({
      id: '1',
      familyName: 'Mustermann',
      givenName: 'Max',
      website: '',
      organization: '',
      email: 'max@mustermann.de',
      orcid: '',
      createdByUserId: null,
      createdAt: null,
      updatedAt: null
    })
    platform.contacts.push(contact)
    const missing = {
      contacts: {
        ids: []
      }
    }

    const result = platformWithMetaToPlatformByAddingDummyObjects({
      platform,
      missing
    })

    expect(result).toEqual(platform)
    expect(result.contacts).toEqual([contact])
  })
  it('should add a dummy contact if there are missing data', () => {
    const platform = new Platform()

    const missing = {
      contacts: {
        ids: ['2']
      }
    }

    const newExpectedContact = new Contact()
    newExpectedContact.id = '2'

    const result = platformWithMetaToPlatformByAddingDummyObjects({
      platform,
      missing
    })

    expect(result.contacts).toEqual([newExpectedContact])
  })
  it('should also add a dummy contact if there are contact data - together with the missing', () => {
    const platform = new Platform()
    const contact = Contact.createFromObject({
      id: '1',
      familyName: 'Mustermann',
      givenName: 'Max',
      website: '',
      organization: '',
      email: 'max@mustermann.de',
      orcid: '',
      createdByUserId: null,
      createdAt: null,
      updatedAt: null
    })
    platform.contacts.push(contact)

    const missing = {
      contacts: {
        ids: ['2']
      }
    }

    const newExpectedContact = new Contact()
    newExpectedContact.id = '2'

    const result = platformWithMetaToPlatformByAddingDummyObjects({
      platform,
      missing
    })

    expect(result.contacts).toEqual([contact, newExpectedContact])
  })
})
