# SPDX-FileCopyrightText: 2023
# - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
# - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
#
# SPDX-License-Identifier: EUPL-1.2

"""Test classes for the site usage community entries."""

import copy

from django.conf import settings
from django.test import TestCase
from django.urls import reverse
from django.utils import encoding

from app.models import Community, GlobalProvenance, SiteUsage, SiteUsageCommunity
from app.serializers import SiteUsageCommunitySerializer
from app.tests.Base_class import BaseTestCase


class SiteUsageTestCaseCommunity(TestCase):
    """Test class for the site usage community model."""

    list_url = reverse("siteusagecommunity-list")

    def setUp(self):
        """Set some data up for the tests."""
        gl = GlobalProvenance.objects.create(
            id=1,
            name="test global provenance",
            description="test global provenance description",
            uri="test global provenance uri",
        )
        gl.save()

        co = Community.objects.create(
            id=1,
            term="Test Community",
            definition="Test definition",
            provenance=None,
            provenance_uri=None,
            category="Test",
            note="Test1",
            global_provenance_id=1,
        )
        co.save()

        su = SiteUsage.objects.create(
            id=1,
            term="Test Site Usage Community",
            definition="Test definition",
            provenance=None,
            provenance_uri=None,
            category="Test",
            note="Test1",
            global_provenance_id=1,
            successor_id=None,
        )
        su.save()
        suc = SiteUsageCommunity.objects.create(
            id=1,
            term="Test Site Usage Community",
            abbreviation="TSTC",
            definition="Test definition",
            provenance=None,
            provenance_uri=None,
            note="Test1",
            community_id=1,
            root_id=1,
        )
        suc.save()
        self.detail_url = reverse(
            "siteusagecommunity-detail", kwargs={"pk": SiteUsageCommunity.pk}
        )

    def test_term(self):
        """Test the term property."""
        suco = SiteUsageCommunity.objects.get(id=1)
        self.assertEqual(suco.term, "Test Site Usage Community")

    def test_serializer(self):
        """
        Test the serializer.

        It should just work fine with the given data.
        But the root & the community are both required.
        """
        suco = SiteUsageCommunity.objects.get(id=1)
        data = SiteUsageCommunitySerializer(suco, context={"request": None}).data
        data_without_root = copy.copy(data)
        data_without_root["root"] = None
        data_without_community = copy.copy(data)
        data_without_community["community"] = None

        # delete, so that we don't run into unique constraints
        suco.delete()

        serializer_ok = SiteUsageCommunitySerializer(
            data=data, context={"request": None}
        )
        self.assertTrue(serializer_ok.is_valid())

        # We still don't save it.
        self.assertEqual(0, SiteUsageCommunity.objects.count())

        serializer_without_root = SiteUsageCommunitySerializer(
            data=data_without_root, context={"request": None}
        )
        self.assertFalse(serializer_without_root.is_valid())

        serializer_without_community = SiteUsageCommunitySerializer(
            data=data_without_community, context={"request": None}
        )
        self.assertFalse(serializer_without_community.is_valid())

    def test_root(self):
        """Test the root relationship."""
        suco = SiteUsageCommunity.objects.get(id=1)
        self.assertEqual(suco.root_id, 1)

    def test_community(self):
        """Test the community relationship."""
        suco = SiteUsageCommunity.objects.get(id=1)
        self.assertEqual(suco.community_id, 1)

    def test_get_all(self, url_name=None):
        """Ensure the result has all attributes in 'SiteUsageCommunity'."""
        suco = SiteUsageCommunity.objects.first()
        if not url_name:
            path = self.list_url
        else:
            path = reverse(
                url_name, kwargs={"parent_lookup_community_id": suco.community_id}
            )

        response = self.client.get(path)
        self.assertEqual(response.status_code, 200)

        expected = {
            "links": {
                "first": f"http://testserver{path}?page%5Bnumber%5D=1",
                "last": f"http://testserver{path}?page%5Bnumber%5D=1",
                "next": None,
                "prev": None,
            },
            "data": [
                {
                    "type": "SiteUsageCommunity",
                    "id": encoding.force_str(suco.pk),
                    "attributes": {
                        "term": suco.term,
                        "abbreviation": suco.abbreviation,
                        "definition": suco.definition,
                        "provenance": suco.provenance,
                        "provenance_uri": suco.provenance_uri,
                        "note": suco.note,
                    },
                    "relationships": {
                        "root": {
                            "links": {
                                "self": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "siteusages-communities/1/relationships/root",
                                    ]
                                ),
                                "related": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "siteusages-communities/1/root/",
                                    ]
                                ),
                            },
                            "data": {
                                "type": "SiteUsage",
                                "id": encoding.force_str(suco.root_id),
                            },
                        },
                        "community": {
                            "links": {
                                "self": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "siteusages-communities/1/relationships/community",
                                    ]
                                ),
                                "related": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "siteusages-communities/1/community/",
                                    ]
                                ),
                            },
                            "data": {
                                "type": "Community",
                                "id": encoding.force_str(suco.community_id),
                            },
                        },
                    },
                    "links": {
                        "self": "".join(
                            [
                                f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                "siteusages-communities/1/",
                            ]
                        ),
                    },
                }
            ],
            "meta": {"pagination": {"page": 1, "pages": 1, "count": 1}},
        }
        assert expected == response.json()

    def test_sort(self):
        """Test that the output is sorted."""
        BaseTestCase().sort(self.list_url, self.client)

    def test_community_routes(self):
        """Test the get list endpoint."""
        self.test_get_all(url_name="community-siteusages-list")

    def test_community_filter(self):
        """Test the filter for the community term."""
        response = self.client.get(
            self.list_url, data={"filter[community.term]": "Test Community"}
        )
        self.assertEqual(
            response.status_code, 200, msg=response.content.decode("utf-8")
        )
        dja_response = response.json()
        self.assertEqual(len(dja_response["data"]), 1)

    def test_rootstatus_filter(self):
        """Test the filter for the root entry."""
        response = self.client.get(
            self.list_url, data={"filter[root.status]": "PENDING"}
        )
        self.assertEqual(
            response.status_code, 200, msg=response.content.decode("utf-8")
        )
        dja_response = response.json()
        self.assertEqual(len(dja_response["data"]), 1)
