# SPDX-FileCopyrightText: 2021 - 2024
# - Kotyba Alhaj Taha <kotyba.alhaj-taha@ufz.de>
# - Jannes Breier <jannes.breier@gfz-potsdam.de>
# - Nils Brinckmann <nils.brinckmann@gfz-potsdam.de>
# - Helmholtz Centre Potsdam - GFZ German Research Centre for Geosciences (GFZ, https://www.gfz-potsdam.de)
# - Helmholtz Centre for Environmental Research GmbH - UFZ (UFZ, https://www.ufz.de)
#
# SPDX-License-Identifier: EUPL-1.2

"""Tests for the aggregation type communities."""

import copy

from django.conf import settings
from django.test import TestCase
from django.urls import reverse
from django.utils import encoding

from app.models import (
    AggregationType,
    AggregationTypeCommunity,
    Community,
    GlobalProvenance,
)
from app.serializers import AggregationTypeCommunitySerializer
from app.tests.Base_class import BaseTestCase


class AggregationTypeCommunityTestCase(TestCase):
    """Test case for the aggregation type communities."""

    list_url = reverse("aggregationtypecommunity-list")

    def setUp(self):
        """Set up some test data."""
        gl = GlobalProvenance.objects.create(
            id=1,
            name="test global provenance",
            description="test global provenance description",
            uri="test global provenance uri",
        )
        gl.save()

        co = Community.objects.create(
            id=1,
            term="Test Community",
            definition="Test definition",
            provenance=None,
            provenance_uri=None,
            category="Test",
            note="Test1",
            global_provenance_id=1,
        )
        co.save()

        ag = AggregationType.objects.create(
            id=1,
            term="Test Aggregation Type",
            definition=None,
            provenance=None,
            provenance_uri=None,
            category=None,
            note=None,
            global_provenance_id=1,
            successor_id=None,
        )
        ag.save()

        agco = AggregationTypeCommunity.objects.create(
            id=1,
            term="Test Aggregation Type Community",
            abbreviation="TATC",
            definition=None,
            provenance=None,
            provenance_uri=None,
            note=None,
            community_id=1,
            root_id=1,
        )
        agco.save()

        self.detail_url = reverse(
            "aggregationtypecommunity-detail",
            kwargs={"pk": AggregationTypeCommunity.pk},
        )

    def test_term(self):
        """Test the term property."""
        agco = AggregationTypeCommunity.objects.get(id=1)
        self.assertEqual(agco.term, "Test Aggregation Type Community")

    def test_serializer(self):
        """
        Test the serializer.

        It should just work fine with the given data.
        But the root & the community are both required.
        """
        agco = AggregationTypeCommunity.objects.get(id=1)
        data = AggregationTypeCommunitySerializer(agco, context={"request": None}).data
        data_without_root = copy.copy(data)
        data_without_root["root"] = None
        data_without_community = copy.copy(data)
        data_without_community["community"] = None

        # delete, so that we don't run into unique constraints
        agco.delete()

        serializer_ok = AggregationTypeCommunitySerializer(
            data=data, context={"request": None}
        )
        self.assertTrue(serializer_ok.is_valid())

        # We still don't save it.
        self.assertEqual(0, AggregationTypeCommunity.objects.count())

        serializer_without_root = AggregationTypeCommunitySerializer(
            data=data_without_root, context={"request": None}
        )
        self.assertFalse(serializer_without_root.is_valid())

        serializer_without_community = AggregationTypeCommunitySerializer(
            data=data_without_community, context={"request": None}
        )
        self.assertFalse(serializer_without_community.is_valid())

    def test_root(self):
        """Test the root property."""
        agco = AggregationTypeCommunity.objects.get(id=1)
        self.assertEqual(agco.root_id, 1)

    def test_community(self):
        """Test the community property."""
        agco = AggregationTypeCommunity.objects.get(id=1)
        self.assertEqual(agco.community_id, 1)

    def test_get_all(self, url_name=None):
        """Ensure the result has all attributes in 'AggregationTypeCommunity'."""
        agco = AggregationTypeCommunity.objects.first()
        if not url_name:
            path = self.list_url
        else:
            path = reverse(
                url_name, kwargs={"parent_lookup_community_id": agco.community_id}
            )

        response = self.client.get(path)
        self.assertEqual(response.status_code, 200)

        expected = {
            "links": {
                "first": f"http://testserver{path}?page%5Bnumber%5D=1",
                "last": f"http://testserver{path}?page%5Bnumber%5D=1",
                "next": None,
                "prev": None,
            },
            "data": [
                {
                    "type": "AggregationTypeCommunity",
                    "id": encoding.force_str(agco.pk),
                    "attributes": {
                        "term": agco.term,
                        "abbreviation": agco.abbreviation,
                        "definition": agco.definition,
                        "provenance": agco.provenance,
                        "provenance_uri": agco.provenance_uri,
                        "note": agco.note,
                    },
                    "relationships": {
                        "root": {
                            "links": {
                                "self": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "aggregationtypes-communities/1/relationships/root",
                                    ]
                                ),
                                "related": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "aggregationtypes-communities/1/root/",
                                    ]
                                ),
                            },
                            "data": {
                                "type": "AggregationType",
                                "id": encoding.force_str(agco.root_id),
                            },
                        },
                        "community": {
                            "links": {
                                "self": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "aggregationtypes-communities/1/relationships/community",
                                    ]
                                ),
                                "related": "".join(
                                    [
                                        f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                        "aggregationtypes-communities/1/community/",
                                    ]
                                ),
                            },
                            "data": {
                                "type": "Community",
                                "id": encoding.force_str(agco.community_id),
                            },
                        },
                    },
                    "links": {
                        "self": "".join(
                            [
                                f"http://testserver/{settings.CV_BASE_URL}api/v1/",
                                "aggregationtypes-communities/1/",
                            ]
                        ),
                    },
                }
            ],
            "meta": {"pagination": {"page": 1, "pages": 1, "count": 1}},
        }
        assert expected == response.json()

    def test_sort(self):
        """Test the ordering."""
        BaseTestCase().sort(self.list_url, self.client)

    def test_community_routes(self):
        """Test the payload for the community related payload."""
        self.test_get_all(url_name="community-aggregationtypes-list")

    def test_community_filter(self):
        """Test the filter for a community term."""
        response = self.client.get(
            self.list_url, data={"filter[community.term]": "Test Community"}
        )
        self.assertEqual(
            response.status_code, 200, msg=response.content.decode("utf-8")
        )
        dja_response = response.json()
        self.assertEqual(len(dja_response["data"]), 1)

    def test_rootstatus_filter(self):
        """Test the filter for the root status."""
        response = self.client.get(
            self.list_url, data={"filter[root.status]": "PENDING"}
        )
        self.assertEqual(
            response.status_code, 200, msg=response.content.decode("utf-8")
        )
        dja_response = response.json()
        self.assertEqual(len(dja_response["data"]), 1)
